#! /usr/bin/python3
import os
import sys
import generator_utils as gen

template = """// @{generatedby}@
/* ///////////////////////// The MPI Bugs Initiative ////////////////////////

  Origin: MBI

  Description: @{shortdesc}@
    @{longdesc}@

   Version of MPI: Conforms to MPI 1.1, does not require MPI 2 implementation

BEGIN_MPI_FEATURES
  P2P!basic: Lacking
  P2P!nonblocking: Lacking
  P2P!persistent: Lacking
  COLL!basic: Lacking
  COLL!nonblocking: Lacking
  COLL!persistent: Lacking
  COLL!tools: @{toolfeature}@
  RMA: Lacking
END_MPI_FEATURES

BEGIN_MBI_TESTS
  $ mpirun -np 2 ${EXE}
  | @{outcome}@
  | @{errormsg}@
END_MBI_TESTS
//////////////////////       End of MBI headers        /////////////////// */

#include <mpi.h>
#include <stdio.h>
#include <stdlib.h>

#define buff_size 128

int main(int argc, char **argv) {
  int nprocs = -1;
  int rank = -1;

  MPI_Init(&argc, &argv);
  MPI_Comm_size(MPI_COMM_WORLD, &nprocs);
  MPI_Comm_rank(MPI_COMM_WORLD, &rank);
  printf("Hello from rank %d \\n", rank);

  if (nprocs < 2)
    printf("MBI ERROR: This test needs at least 2 processes to produce a bug!\\n");

   MPI_Comm newcom;
   int dims[2], periods[2], coords[2];
   int source, dest;
   dims[0] = 2;
   dims[1] = 1;
   periods[0] = 1;
   periods[1] = 1;

   @{change_dims}@

   MPI_Cart_create(MPI_COMM_WORLD, 2, dims, periods, 0, &newcom); /* create a cartesian communicator */

   @{change_com}@

   @{init}@
   @{operation}@ /* MBIERROR2 */
   @{fini}@

   if (newcom != MPI_COMM_NULL)
     MPI_Comm_free(&newcom);

  MPI_Finalize();
  printf("Rank %d finished normally\\n", rank);
  return 0;
}
"""

for c in gen.tcoll4topo:
    patterns = {}
    patterns = {'c': c}
    patterns['generatedby'] = f'DO NOT EDIT: this file was generated by {os.path.basename(sys.argv[0])}. DO NOT EDIT.'
    patterns['toolfeature'] = 'Yes' if c in gen.tcoll4topo else 'Lacking'
    patterns['c'] = c
    patterns['init'] = gen.init[c]("1")
    patterns['fini'] = gen.fini[c]("1")
    patterns['operation'] = gen.operation[c]("1")
    patterns['change_dims'] = '/* No error injected here */'

    # Generate the correct code
    replace = patterns.copy()
    replace['shortdesc'] = 'Function @{c}@ with correct arguments'
    replace['longdesc'] = f'All ranks in comm call {c} with correct arguments'
    replace['outcome'] = 'OK'
    replace['errormsg'] = ''
    replace['change_com'] = '/* No error injected here */'
    replace['change_dims'] = '/* No error injected here */'
    gen.make_file(template, f'InvalidParam_{c}_ok.c', replace)

    # Generate the incorrect code
    replace = patterns.copy()
    replace['shortdesc'] = 'The code tries to get cartesian information of MPI_COMM_WORLD.'
    replace['longdesc'] = 'The code creates a cartesian communicator, and tries to get cartesian information of MPI_COMM_WORLD.'
    replace['outcome'] = 'ERROR: InvalidCommunicator'
    replace['errormsg'] = 'Invalid Communicator in a collective. @{c}@ at @{filename}@:@{line:MBIERROR2}@ tries to get cartesian information of MPI_COMM_WORLD.'
    replace['change_com'] = 'newcom = MPI_COMM_WORLD; /* MBIERROR1 */'
    gen.make_file(template, f'InvalidParam_Com_{c}_nok.c', replace)

    # Generate the code with newcom=MPI_COMM_NULL
    replace = patterns.copy()
    replace['shortdesc'] = 'Function @{c}@ called with comm=MPI_COMM_NULL'
    replace['longdesc'] = 'Function @{c}@ called with comm=MPI_COMM_NULL'
    replace['outcome'] = 'ERROR: InvalidCommunicator'
    replace['errormsg'] = 'Invalid communicator. @{c}@ at @{filename}@:@{line:MBIERROR2}@ has MPI_COMM_NULL as a communicator.'
    replace['change_com'] = 'newcom = MPI_COMM_NULL; /* MBIERROR1 */'
    gen.make_file(template, f'InvalidParam_ComNull_{c}_nok.c', replace)

    # Generate the code with invalid dimension
    replace = patterns.copy()
    replace['shortdesc'] = 'Creates a cartesian communicator with a negative entry in the dims attribute'
    replace['longdesc'] = 'Creates a cartesian communicator with a negative entry in the dims attribute, which is a usage error'
    replace['outcome'] = 'ERROR: InvalidOtherArg'
    replace['errormsg'] = 'Invalid Argument. MPI_Cart_create has invalid dimensions.'
    replace['change_com'] = ""
    replace['change_dims'] = 'dims[0] = -2; dims[1] = -1; /* MBIERROR1 */'
    gen.make_file(template, 'InvalidParam_Dim_MPI_Cart_create_nok.c', replace)
