/*
   This file is part of TALER
   Copyright (C) 2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_insert_report.c
 * @brief Implementation of the insert_report function for Postgres
 * @author Christian Grothoff
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_insert_report.h"
#include "pg_helper.h"


enum GNUNET_DB_QueryStatus
TMH_PG_insert_report (
  void *cls,
  const char *instance_id,
  const char *report_program_section,
  const char *report_description,
  const char *mime_type,
  const char *data_source,
  const char *target_address,
  struct GNUNET_TIME_Relative frequency,
  struct GNUNET_TIME_Relative frequency_shift,
  uint64_t *report_id)
{
  struct PostgresClosure *pg = cls;
  struct TALER_MERCHANT_ReportToken report_token;
  struct GNUNET_TIME_Timestamp start;
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (instance_id),
    GNUNET_PQ_query_param_string (report_program_section),
    GNUNET_PQ_query_param_string (report_description),
    GNUNET_PQ_query_param_string (mime_type),
    GNUNET_PQ_query_param_auto_from_type (&report_token),
    GNUNET_PQ_query_param_string (data_source),
    GNUNET_PQ_query_param_string (target_address),
    GNUNET_PQ_query_param_relative_time (&frequency),
    GNUNET_PQ_query_param_relative_time (&frequency_shift),
    GNUNET_PQ_query_param_timestamp (&start),
    GNUNET_PQ_query_param_end
  };
  struct GNUNET_PQ_ResultSpec rs[] = {
    GNUNET_PQ_result_spec_uint64 ("report_serial",
                                  report_id),
    GNUNET_PQ_result_spec_end
  };

  GNUNET_CRYPTO_random_block (GNUNET_CRYPTO_QUALITY_NONCE,
                              &report_token,
                              sizeof (report_token));
  start =
    GNUNET_TIME_absolute_to_timestamp (
      GNUNET_TIME_absolute_add (
        GNUNET_TIME_absolute_round_down (GNUNET_TIME_absolute_get (),
                                         frequency),
        GNUNET_TIME_relative_add (frequency,
                                  frequency_shift)));

  check_connection (pg);
  PREPARE (pg,
           "insert_report",
           "INSERT INTO merchant_reports"
           "(merchant_serial"
           ",report_program_section"
           ",report_description"
           ",mime_type"
           ",report_token"
           ",data_source"
           ",target_address"
           ",frequency"
           ",frequency_shift"
           ",next_transmission)"
           " SELECT merchant_serial, $2, $3, $4, $5,"
           "   $6, $7, $8, $9, $10"
           " FROM merchant_instances"
           " WHERE merchant_id=$1"
           " ON CONFLICT DO NOTHING;");

  return GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                   "insert_report",
                                                   params,
                                                   rs);
}
