#########################################################################
#             GNU HEALTH HOSPITAL MANAGEMENT - GTK CLIENT               #
#                      https://www.gnuhealth.org                        #
#########################################################################
#       The GNUHealth HMIS client based on the Tryton GTK Client        #
#########################################################################
#
# SPDX-FileCopyrightText: 2008-2024 The Tryton Community <info@tryton.org>
# SPDX-FileCopyrightText: 2017-2024 GNU Health Community <info@gnuhealth.org>
#
# SPDX-License-Identifier: GPL-3.0-or-later

# This file is part of GNU Health.  The COPYRIGHT file at the top level of
# this repository contains the full copyright notices and license terms.
"Preference"
import copy
import gettext

from gi.repository import Gdk, Gtk

import gnuhealth.rpc as rpc
from gnuhealth.common import IconFactory, RPCException, RPCExecute
from gnuhealth.common.underline import set_underline
from gnuhealth.config import GNUHEALTH_ICON
from gnuhealth.gui import Main
from gnuhealth.gui.window.nomodal import NoModal
from gnuhealth.gui.window.view_form.screen import Screen

_ = gettext.gettext


class Preference(NoModal):
    "Preference window"

    def __init__(self, user, callback):
        NoModal.__init__(self)
        self.callback = callback
        self.win = Gtk.Dialog(
            title=_('Preferences'), transient_for=self.parent,
            destroy_with_parent=True)
        Main().add_window(self.win)
        self.win.set_position(Gtk.WindowPosition.CENTER_ON_PARENT)
        self.win.set_icon(GNUHEALTH_ICON)

        self.accel_group = Gtk.AccelGroup()
        self.win.add_accel_group(self.accel_group)

        self.but_cancel = self.win.add_button(
            set_underline(_("Cancel")), Gtk.ResponseType.CANCEL)
        self.but_cancel.set_image(
            IconFactory.get_image('gnuhealth-cancel', Gtk.IconSize.BUTTON))
        self.but_cancel.set_always_show_image(True)
        self.but_ok = self.win.add_button(
            set_underline(_("OK")), Gtk.ResponseType.OK)
        self.but_ok.set_image(
            IconFactory.get_image('gnuhealth-ok', Gtk.IconSize.BUTTON))
        self.but_ok.set_always_show_image(True)
        self.but_ok.add_accelerator(
            'clicked', self.accel_group, Gdk.KEY_Return,
            Gdk.ModifierType.CONTROL_MASK, Gtk.AccelFlags.VISIBLE)

        self.win.set_default_response(Gtk.ResponseType.OK)
        self.win.connect('response', self.response)

        try:
            view = RPCExecute('model', 'res.user',
                              'get_preferences_fields_view')
        except RPCException:
            self.win.destroy()
            self.win = None
            return

        title = Gtk.Label(label=_('Edit User Preferences'))
        title.show()
        self.win.vbox.pack_start(title, expand=False, fill=True, padding=0)
        self.screen = Screen('res.user', mode=[])
        # Reset readonly set automaticly by MODELACCESS
        self.screen.readonly = False
        self.screen.group.readonly = False
        self.screen.group.skip_model_access = True
        self.screen.add_view(view)
        self.screen.switch_view()
        self.screen.new(default=False)

        try:
            preferences = RPCExecute('model', 'res.user', 'get_preferences',
                                     False)
        except RPCException:
            self.win.destroy()
            self.win = None
            return
        self.screen.current_record.cancel()
        self.screen.current_record.set(preferences)
        self.screen.current_record.id = rpc._USER
        self.screen.current_record.validate(softvalidation=True)
        self.screen.display(set_cursor=True)

        self.screen.widget.show()
        self.win.vbox.pack_start(
            self.screen.widget, expand=True, fill=True, padding=0)
        self.win.set_title(_('Preference'))

        self.win.set_default_size(*self.default_size())

        self.register()
        self.win.show()

    def response(self, win, response_id):
        if response_id == Gtk.ResponseType.OK:
            if self.screen.current_record.validate():
                vals = copy.copy(self.screen.get())
                try:
                    RPCExecute('model', 'res.user', 'set_preferences', vals)
                except RPCException:
                    return
                rpc.context_reset()
        self.parent.present()
        self.destroy()
        self.callback()

    def destroy(self):
        self.screen.destroy()
        self.win.destroy()
        NoModal.destroy(self)
