/*
 * Decompiled with CFR 0.152.
 */
package org.torproject.metrics.exonerator;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.LineNumberReader;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TimeZone;
import java.util.TreeMap;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.torproject.descriptor.Descriptor;
import org.torproject.descriptor.DescriptorCollector;
import org.torproject.descriptor.DescriptorReader;
import org.torproject.descriptor.DescriptorSourceFactory;
import org.torproject.descriptor.ExitList;
import org.torproject.descriptor.NetworkStatusEntry;
import org.torproject.descriptor.RelayNetworkStatusConsensus;
import org.torproject.descriptor.UnparseableDescriptor;

public class ExoneraTorDatabaseImporter {
    private static final Logger logger = LoggerFactory.getLogger(ExoneraTorDatabaseImporter.class);
    private static String jdbcString;
    private static File importDirectory;
    private static Connection connection;
    private static CallableStatement insertStatusentryStatement;
    private static CallableStatement insertExitlistentryStatement;
    private static SortedMap<String, Long> lastImportHistory;
    private static SortedMap<String, Long> nextImportHistory;
    private static File parseHistoryFile;

    public static void main(String[] args) {
        Locale.setDefault(Locale.US);
        TimeZone.setDefault(TimeZone.getTimeZone("UTC"));
        logger.info("Starting ExoneraTor database importer.");
        ExoneraTorDatabaseImporter.readConfiguration();
        ExoneraTorDatabaseImporter.openDatabaseConnection();
        ExoneraTorDatabaseImporter.prepareDatabaseStatements();
        ExoneraTorDatabaseImporter.createLockFile();
        logger.info("Fetching descriptors from CollecTor.");
        ExoneraTorDatabaseImporter.fetchDescriptors();
        logger.info("Importing descriptors into the database.");
        ExoneraTorDatabaseImporter.readImportHistoryToMemory();
        ExoneraTorDatabaseImporter.parseDescriptors();
        ExoneraTorDatabaseImporter.writeImportHistoryToDisk();
        ExoneraTorDatabaseImporter.closeDatabaseConnection();
        ExoneraTorDatabaseImporter.deleteLockFile();
        logger.info("Terminating ExoneraTor database importer.");
    }

    private static void readConfiguration() {
        File configFile = new File("config");
        if (!configFile.exists()) {
            logger.error("Could not find configuration file {}. Make sure that this file exists. Exiting.", (Object)configFile.getAbsoluteFile());
            System.exit(1);
        }
        try (BufferedReader br = new BufferedReader(new FileReader(configFile));){
            String line;
            while ((line = br.readLine()) != null) {
                if (line.startsWith("ExoneraTorDatabaseJdbc")) {
                    jdbcString = line.split(" ")[1];
                    continue;
                }
                if (!line.startsWith("ExoneraTorImportDirectory")) continue;
                importDirectory = new File(line.split(" ")[1]);
            }
        }
        catch (IOException e) {
            logger.error("Caught an I/O exception while reading configuration file {}. Make sure that this file is readable. Exiting.", (Object)configFile.getAbsoluteFile(), (Object)e);
            System.exit(1);
        }
        catch (ArrayIndexOutOfBoundsException e) {
            logger.error("Found invalid entry in configuration file {} containing fewer than 2 space-separated parts. Fix that line. Exiting.", (Object)configFile.getAbsoluteFile());
            System.exit(1);
        }
        if (null == jdbcString || null == importDirectory) {
            logger.error("Missing at least one mandatory line in configuration file {}. Be sure to configure ExoneraTorDatabaseJdbc and ExoneraTorImportDirectory. Exiting.", (Object)configFile.getAbsoluteFile());
            System.exit(1);
        }
        logger.debug("Read configuration file {}.", (Object)configFile.getAbsoluteFile());
    }

    private static void openDatabaseConnection() {
        try {
            connection = DriverManager.getConnection(jdbcString);
        }
        catch (SQLException e) {
            logger.error("Caught an SQL exception while connecting to the database. Make sure that the database exists and that the configured JDBC string is correct.", (Throwable)e);
            System.exit(1);
        }
        logger.debug("Connected to the database.");
    }

    private static void prepareDatabaseStatements() {
        try {
            insertStatusentryStatement = connection.prepareCall("{call insert_statusentry_oraddress(?, ?, ?, ?, ?, ?)}");
            insertExitlistentryStatement = connection.prepareCall("{call insert_exitlistentry_exitaddress(?, ?, ?, ?)}");
        }
        catch (SQLException e) {
            logger.error("Caught an SQL exception while preparing callable statements for importing data into the database. Make sure that the configured database user has permissions to insert data. Also make sure that the database uses the correct database schema.", (Throwable)e);
            System.exit(1);
        }
    }

    private static void createLockFile() {
        File lockFile = new File("exonerator-lock");
        if (lockFile.exists()) {
            try (BufferedReader br = new BufferedReader(new FileReader(lockFile));){
                Instant runStarted = Instant.ofEpochMilli(Long.parseLong(br.readLine()));
                if (runStarted.plus(Duration.ofHours(6L)).compareTo(Instant.now()) >= 0) {
                    logger.error("Lock file {} is less than 6 hours old. Either make sure that there are no other ExoneraTor database importers running and manually delete that file, or wait until the file is 6 hours old when it will be overwritten automatically. Exiting.", (Object)lockFile.getAbsoluteFile());
                    System.exit(1);
                } else {
                    logger.warn("Lock file {} is at least 6 hours old. Overwriting and continuing with the database import.", (Object)lockFile.getAbsoluteFile());
                }
            }
            catch (IOException e) {
                logger.error("Caught an I/O exception when reading existing lock file {}. Make sure that this file is readable. Exiting.", (Object)lockFile.getAbsoluteFile(), (Object)e);
                System.exit(1);
            }
        }
        try (BufferedWriter bw = new BufferedWriter(new FileWriter(lockFile));){
            bw.append(String.valueOf(System.currentTimeMillis())).append("\n");
        }
        catch (IOException e) {
            logger.error("Caught an I/O exception when creating lock file {}. Make sure that the parent directory exists and that the user running the ExoneraTor database importer has permissions to create the lock file. Exiting.", (Object)lockFile.getAbsoluteFile(), (Object)e);
            System.exit(1);
        }
        logger.debug("Created lock file {}.", (Object)lockFile.getAbsoluteFile());
    }

    private static void fetchDescriptors() {
        DescriptorCollector collector = DescriptorSourceFactory.createDescriptorCollector();
        collector.collectDescriptors("https://collector.torproject.org", new String[]{"/recent/relay-descriptors/consensuses/", "/recent/exit-lists/"}, 0L, importDirectory, true);
    }

    private static void readImportHistoryToMemory() {
        if (parseHistoryFile.exists()) {
            try (LineNumberReader lnr = new LineNumberReader(new FileReader(parseHistoryFile));){
                String line;
                while ((line = lnr.readLine()) != null) {
                    Long lastModified = null;
                    String filename = null;
                    String[] parts = line.split(",");
                    if (parts.length == 2) {
                        try {
                            lastModified = Long.parseLong(parts[0]);
                            filename = parts[1];
                        }
                        catch (NumberFormatException numberFormatException) {
                            // empty catch block
                        }
                    }
                    if (null == lastModified || null == filename) {
                        logger.warn("Read a corrupt entry in line {} of parse history file {}. Ignoring the parse history file entirely and moving on by parsing all descriptors in {}.", new Object[]{lnr.getLineNumber(), parseHistoryFile.getAbsoluteFile(), importDirectory.getAbsoluteFile()});
                        lastImportHistory.clear();
                        return;
                    }
                    lastImportHistory.put(filename, lastModified);
                }
            }
            catch (IOException e) {
                logger.warn("Caught an I/O exception while reading parse history file {}. Ignoring the parse history file entirely and moving on by parsing all descriptors in {}.", new Object[]{parseHistoryFile.getAbsoluteFile(), importDirectory.getAbsoluteFile(), e});
                lastImportHistory.clear();
                return;
            }
            logger.debug("Read parse history file {} and extracted {} entries.", (Object)parseHistoryFile.getAbsoluteFile(), (Object)lastImportHistory.size());
        } else {
            logger.debug("Not reading parse history file {}, because it does not yet exist.", (Object)parseHistoryFile.getAbsoluteFile());
        }
    }

    private static void parseDescriptors() {
        DescriptorReader descriptorReader = DescriptorSourceFactory.createDescriptorReader();
        descriptorReader.setMaxDescriptorsInQueue(20);
        descriptorReader.setExcludedFiles(lastImportHistory);
        int parsedConsensuses = 0;
        int parsedExitLists = 0;
        int unparseableDescriptors = 0;
        for (Descriptor descriptor : descriptorReader.readDescriptors(new File[]{importDirectory})) {
            if (descriptor instanceof RelayNetworkStatusConsensus) {
                ExoneraTorDatabaseImporter.parseConsensus((RelayNetworkStatusConsensus)descriptor);
                ++parsedConsensuses;
                continue;
            }
            if (descriptor instanceof ExitList) {
                ExoneraTorDatabaseImporter.parseExitList((ExitList)descriptor);
                ++parsedExitLists;
                continue;
            }
            if (!(descriptor instanceof UnparseableDescriptor)) continue;
            logger.debug("Found descriptor in {} to be unparseable. Check the descriptor parse exception and/or descriptor file for details. Skipping.", (Object)descriptor.getDescriptorFile().getAbsoluteFile(), (Object)((UnparseableDescriptor)descriptor).getDescriptorParseException());
            ++unparseableDescriptors;
        }
        if (unparseableDescriptors > 0) {
            logger.warn("Found {} descriptors in {} to be unparseable and skipped them. Check the debug-level logs and/or descriptor files for details. If this happened due to a bug in the parsing code, manually delete the parse history file {} and run the database importer again. Continuing.", new Object[]{unparseableDescriptors, importDirectory.getAbsoluteFile(), parseHistoryFile.getAbsoluteFile()});
        }
        nextImportHistory.putAll(descriptorReader.getExcludedFiles());
        nextImportHistory.putAll(descriptorReader.getParsedFiles());
        logger.debug("Read {} consensuses and {} exit lists from {}.", new Object[]{parsedConsensuses, parsedExitLists, importDirectory.getAbsoluteFile()});
    }

    private static void parseConsensus(RelayNetworkStatusConsensus consensus) {
        Instant beforeParsingConsensus = Instant.now();
        LocalDateTime validAfter = LocalDateTime.ofInstant(Instant.ofEpochMilli(consensus.getValidAfterMillis()), ZoneOffset.UTC);
        int importedStatusEntries = 0;
        for (NetworkStatusEntry entry : consensus.getStatusEntries().values()) {
            if (!entry.getFlags().contains("Running")) continue;
            String fingerprintBase64 = null;
            try {
                fingerprintBase64 = Base64.encodeBase64String((byte[])Hex.decodeHex((char[])entry.getFingerprint().toCharArray())).replace("=", "");
            }
            catch (DecoderException e) {
                logger.error("Caught a decoder exception while converting hex fingerprint {} found in consensus with valid-after time {} to base64. This looks like a bug. Exiting.", new Object[]{entry.getFingerprint(), validAfter, e});
                System.exit(1);
            }
            String nickname = entry.getNickname();
            Boolean exit = null;
            if (null != entry.getDefaultPolicy() && null != entry.getPortList()) {
                exit = "accept".equals(entry.getDefaultPolicy()) || !"1-65535".equals(entry.getPortList());
            }
            HashSet<String> orAddresses = new HashSet<String>();
            orAddresses.add(entry.getAddress());
            for (String orAddressAndPort : entry.getOrAddresses()) {
                orAddresses.add(orAddressAndPort.substring(0, orAddressAndPort.lastIndexOf(58)));
            }
            ExoneraTorDatabaseImporter.importStatusentry(validAfter, fingerprintBase64, nickname, exit, orAddresses);
            ++importedStatusEntries;
        }
        logger.debug("Parsed consensus with valid-after time {} and imported {} status entries with the Running flag into the database in {}.", new Object[]{validAfter, importedStatusEntries, Duration.between(beforeParsingConsensus, Instant.now())});
    }

    private static void importStatusentry(LocalDateTime validAfter, String fingerprintBase64, String nickname, Boolean exit, Set<String> orAddresses) {
        try {
            for (String orAddress : orAddresses) {
                insertStatusentryStatement.clearParameters();
                insertStatusentryStatement.setObject(1, (Object)validAfter);
                insertStatusentryStatement.setString(2, fingerprintBase64);
                if (!orAddress.contains(":")) {
                    insertStatusentryStatement.setString(3, orAddress);
                    String[] addressParts = orAddress.split("\\.");
                    byte[] address24Bytes = new byte[]{(byte)Integer.parseInt(addressParts[0]), (byte)Integer.parseInt(addressParts[1]), (byte)Integer.parseInt(addressParts[2])};
                    String orAddress24 = Hex.encodeHexString((byte[])address24Bytes);
                    insertStatusentryStatement.setString(4, orAddress24);
                } else {
                    String[] parts;
                    StringBuilder addressHex = new StringBuilder();
                    int start = orAddress.startsWith("[::") ? 2 : 1;
                    int end = orAddress.length() - (orAddress.endsWith("::]") ? 2 : 1);
                    for (String part : parts = orAddress.substring(start, end).split(":", -1)) {
                        if (part.length() == 0) {
                            addressHex.append("x");
                            continue;
                        }
                        if (part.length() <= 4) {
                            addressHex.append(String.format("%4s", part));
                            continue;
                        }
                        addressHex = null;
                        break;
                    }
                    String orAddress24 = null;
                    if (addressHex != null) {
                        String addressHexString = addressHex.toString();
                        if (!(addressHexString = addressHexString.replaceFirst("x", String.format("%" + (33 - addressHexString.length()) + "s", "0"))).contains("x") && addressHexString.length() == 32) {
                            orAddress24 = addressHexString.replace(" ", "0").toLowerCase().substring(0, 6);
                        }
                    }
                    if (orAddress24 != null) {
                        insertStatusentryStatement.setString(3, orAddress.replaceAll("[\\[\\]]", ""));
                        insertStatusentryStatement.setString(4, orAddress24);
                    } else {
                        logger.error("Unable to parse IPv6 address {} found in status entry with base64-encoded fingerprint {} in consensus with valid-after time {}. This is likely a bug. Exiting.", new Object[]{orAddress, fingerprintBase64, validAfter});
                        System.exit(1);
                    }
                }
                insertStatusentryStatement.setString(5, nickname);
                insertStatusentryStatement.setBoolean(6, (boolean)exit);
                insertStatusentryStatement.execute();
            }
        }
        catch (SQLException e) {
            logger.error("Caught an SQL exception while importing status entry with base64-encoded fingerprint {} and valid-after time {}. Check the exception for details. Exiting.", new Object[]{fingerprintBase64, validAfter, e});
            System.exit(1);
        }
    }

    private static void parseExitList(ExitList exitList) {
        Instant beforeParsingExitList = Instant.now();
        LocalDateTime downloaded = LocalDateTime.ofInstant(Instant.ofEpochMilli(exitList.getDownloadedMillis()), ZoneOffset.UTC);
        int importedExitListEntries = 0;
        for (ExitList.Entry entry : exitList.getEntries()) {
            for (Map.Entry e : entry.getExitAddresses().entrySet()) {
                String fingerprintBase64 = null;
                try {
                    fingerprintBase64 = Base64.encodeBase64String((byte[])Hex.decodeHex((char[])entry.getFingerprint().toCharArray())).replace("=", "");
                }
                catch (DecoderException ex) {
                    logger.error("Caught a decoder exception while converting hex fingerprint {} found in exit list downloaded (by CollecTor) at {} to base64. This looks like a bug. Exiting.", new Object[]{entry.getFingerprint(), downloaded, ex});
                    System.exit(1);
                }
                String exitAddress = (String)e.getKey();
                String[] exitAddressParts = exitAddress.split("\\.");
                byte[] exitAddress24Bytes = new byte[]{(byte)Integer.parseInt(exitAddressParts[0]), (byte)Integer.parseInt(exitAddressParts[1]), (byte)Integer.parseInt(exitAddressParts[2])};
                String exitAddress24 = Hex.encodeHexString((byte[])exitAddress24Bytes);
                LocalDateTime scanned = LocalDateTime.ofInstant(Instant.ofEpochMilli((Long)e.getValue()), ZoneOffset.UTC);
                ExoneraTorDatabaseImporter.importExitlistentry(fingerprintBase64, exitAddress24, exitAddress, scanned);
                ++importedExitListEntries;
            }
        }
        logger.debug("Parsed exit list downloaded (by CollecTor) at {} and imported {} exit list entries into the database in {}.", new Object[]{downloaded, importedExitListEntries, Duration.between(beforeParsingExitList, Instant.now())});
    }

    private static void importExitlistentry(String fingerprintBase64, String exitAddress24, String exitAddress, LocalDateTime scanned) {
        try {
            insertExitlistentryStatement.clearParameters();
            insertExitlistentryStatement.setString(1, fingerprintBase64);
            insertExitlistentryStatement.setString(2, exitAddress);
            insertExitlistentryStatement.setString(3, exitAddress24);
            insertExitlistentryStatement.setObject(4, (Object)scanned);
            insertExitlistentryStatement.execute();
        }
        catch (SQLException e) {
            logger.error("Caught an SQL exception while importing exit list entry with base64-encoded fingerprint {}, exit address {}, and scan time {}. Check the exception for details. Exiting.", new Object[]{fingerprintBase64, exitAddress, scanned, e});
            System.exit(1);
        }
    }

    private static void writeImportHistoryToDisk() {
        if (parseHistoryFile.getParentFile().mkdirs()) {
            logger.debug("Created parent directory of parse history file {}.", (Object)parseHistoryFile.getAbsoluteFile());
        }
        try (BufferedWriter bw = new BufferedWriter(new FileWriter(parseHistoryFile));){
            for (Map.Entry<String, Long> historyEntry : nextImportHistory.entrySet()) {
                bw.write(historyEntry.getValue() + "," + historyEntry.getKey() + "\n");
            }
        }
        catch (IOException e) {
            logger.warn("Caught an I/O exception while writing parse history file {}. The next execution might not be able to read this parse history and will parse all files in {}. Moving on, because there is nothing we can do about it.", new Object[]{parseHistoryFile, importDirectory.getAbsoluteFile(), e});
        }
        logger.debug("Wrote parse history file {}.", (Object)parseHistoryFile.getAbsoluteFile());
    }

    private static void closeDatabaseConnection() {
        try {
            connection.close();
            logger.debug("Disconnected from database.");
        }
        catch (SQLException e) {
            logger.warn("Caught an SQL exception while disconnecting from the database. Check the exception for details and ideally log into the database manually to check that everything has been imported correctly. Ignoring, because we were going to terminate anyway.", (Throwable)e);
        }
    }

    private static void deleteLockFile() {
        Path lockFile = Paths.get("exonerator-lock", new String[0]);
        try {
            Files.delete(lockFile);
            logger.debug("Deleted lock file {}.", (Object)lockFile);
        }
        catch (IOException e) {
            logger.warn("Caught an I/O exception while deleting lock file {}. This might prevent future executions from running until the lock file is 6 hours old and overwritten, provided that the file can be overwritten. Moving on, because we cannot do anything about it.", (Object)lockFile, (Object)e);
        }
    }

    static {
        lastImportHistory = new TreeMap<String, Long>();
        nextImportHistory = new TreeMap<String, Long>();
        parseHistoryFile = new File("stats", "exonerator-import-history");
    }
}

