/*
 * Decompiled with CFR 0.152.
 */
package org.torproject.metrics.stats.clients;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.LineNumberReader;
import java.nio.file.Path;
import java.time.LocalDate;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.stream.Collectors;
import org.apache.commons.math3.distribution.NormalDistribution;
import org.apache.commons.math3.distribution.PoissonDistribution;
import org.apache.commons.math3.stat.descriptive.moment.Mean;
import org.apache.commons.math3.stat.descriptive.moment.StandardDeviation;
import org.apache.commons.math3.stat.descriptive.rank.Percentile;
import org.torproject.metrics.stats.clients.Main;

public class Detector {
    private static final Path INPUT_PATH = new File(Main.baseDir, "stats/userstats.csv").toPath();
    private static final Path OUTPUT_PATH = new File(Main.baseDir, "stats/clients.csv").toPath();
    private static final int NUM_LARGEST_LOCATIONS = 50;
    private static final int INTERV = 7;

    public void detect() throws IOException {
        SortedMap<ClientsKey, ClientsEstimates> estimates = Detector.readInputFile();
        Set<String> largestLocations = Detector.findLargestLocations(estimates);
        Map<LocalDate, List<Double>> ratios = Detector.computeRatiosOfLargestLocations(estimates, largestLocations);
        SortedMap<LocalDate, List<Double>> ratiosWithoutOutliers = Detector.removeOutliers(ratios);
        SortedMap<ClientsKey, ClientsRanges> ranges = Detector.computeRanges(estimates, ratiosWithoutOutliers);
        Detector.writeOutputFile(estimates, ranges);
    }

    private static SortedMap<ClientsKey, ClientsEstimates> readInputFile() throws IOException {
        TreeMap<ClientsKey, ClientsEstimates> estimates = new TreeMap<ClientsKey, ClientsEstimates>();
        File inputFile = INPUT_PATH.toFile();
        if (!inputFile.exists()) {
            throw new IOException(String.format("Input file %s does not exist.", inputFile));
        }
        try (LineNumberReader lnr = new LineNumberReader(new FileReader(inputFile));){
            String line = lnr.readLine();
            if (!"date,node,country,transport,version,frac,users".equals(line)) {
                throw new IOException(String.format("Unable to read input file %s with unrecognized header line '%s'. Not running detector.", inputFile, line));
            }
            while ((line = lnr.readLine()) != null) {
                ClientsKey key = null;
                ClientsEstimates value = null;
                boolean invalidLine = false;
                String[] lineParts = line.split(",");
                if (lineParts.length == 7) {
                    try {
                        LocalDate date = LocalDate.parse(lineParts[0]);
                        boolean nodeIsRelay = false;
                        if ("relay".equals(lineParts[1])) {
                            nodeIsRelay = true;
                        } else if (!"bridge".equals(lineParts[1])) {
                            invalidLine = true;
                        }
                        String country = lineParts[2].replaceAll("\"", "");
                        String transport = lineParts[3].replaceAll("\"", "");
                        String version = lineParts[4].replaceAll("\"", "");
                        key = new ClientsKey(date, nodeIsRelay, country, transport, version);
                    }
                    catch (DateTimeParseException e) {
                        invalidLine = true;
                    }
                    try {
                        int frac = Integer.parseInt(lineParts[5]);
                        int clients = Integer.parseInt(lineParts[6]);
                        value = new ClientsEstimates(clients, frac);
                    }
                    catch (NumberFormatException e) {
                        invalidLine = true;
                    }
                } else {
                    invalidLine = true;
                }
                if (invalidLine) {
                    throw new IOException(String.format("Invalid line %d '%s' in input file %s.", lnr.getLineNumber(), line, inputFile));
                }
                estimates.put(key, value);
            }
        }
        return estimates;
    }

    private static Set<String> findLargestLocations(SortedMap<ClientsKey, ClientsEstimates> clients) throws IOException {
        LocalDate lastKnownDate = clients.keySet().stream().filter(c -> ((ClientsKey)c).nodeIsRelay).map(c -> ((ClientsKey)c).date).max(LocalDate::compareTo).orElseThrow(() -> new IOException("Unable to find maximum date. Was the input file empty or otherwise corrupt?"));
        return clients.entrySet().stream().filter(c -> lastKnownDate.equals(((ClientsKey)c.getKey()).date)).filter(c -> ((ClientsKey)c.getKey()).nodeIsRelay).filter(c -> !"".equals(((ClientsKey)c.getKey()).country)).filter(c -> !"??".equals(((ClientsKey)c.getKey()).country)).sorted((c1, c2) -> Integer.compare(((ClientsEstimates)c2.getValue()).clients, ((ClientsEstimates)c1.getValue()).clients)).map(c -> ((ClientsKey)c.getKey()).country).limit(50L).collect(Collectors.toSet());
    }

    private static Map<LocalDate, List<Double>> computeRatiosOfLargestLocations(SortedMap<ClientsKey, ClientsEstimates> estimates, Set<String> largestLocations) {
        HashMap<LocalDate, List<Double>> ratios = new HashMap<LocalDate, List<Double>>();
        for (Map.Entry<ClientsKey, ClientsEstimates> numerator : estimates.entrySet()) {
            ClientsEstimates denominator;
            if (!numerator.getKey().nodeIsRelay || !largestLocations.contains(numerator.getKey().country) || null == (denominator = (ClientsEstimates)estimates.get(new ClientsKey(numerator.getKey().date.minusDays(7L), true, numerator.getKey().country))) || denominator.clients == 0) continue;
            if (!ratios.containsKey(numerator.getKey().date)) {
                ratios.put(numerator.getKey().date, new ArrayList());
            }
            ((List)ratios.get(numerator.getKey().date)).add((double)numerator.getValue().clients / (double)denominator.clients);
        }
        return ratios;
    }

    private static SortedMap<LocalDate, List<Double>> removeOutliers(Map<LocalDate, List<Double>> ratios) {
        TreeMap<LocalDate, List<Double>> ratiosWithoutOutliers = new TreeMap<LocalDate, List<Double>>();
        for (Map.Entry<LocalDate, List<Double>> e : ratios.entrySet()) {
            double[] values = e.getValue().stream().mapToDouble(Double::doubleValue).toArray();
            Percentile percentile = new Percentile().withEstimationType(Percentile.EstimationType.R_7);
            percentile.setData(values);
            double median = percentile.evaluate(50.0);
            double firstQuarter = percentile.evaluate(25.0);
            double thirdQuarter = percentile.evaluate(75.0);
            double interQuartileRange = thirdQuarter - firstQuarter;
            ArrayList<Double> valuesWithoutOutliers = new ArrayList<Double>();
            for (double value : values) {
                if (!(value > median - 4.0 * interQuartileRange) || !(value < median + 4.0 * interQuartileRange)) continue;
                valuesWithoutOutliers.add(value);
            }
            if (valuesWithoutOutliers.size() < 8) continue;
            LocalDate date = e.getKey();
            ratiosWithoutOutliers.put(date, valuesWithoutOutliers);
        }
        return ratiosWithoutOutliers;
    }

    private static SortedMap<ClientsKey, ClientsRanges> computeRanges(SortedMap<ClientsKey, ClientsEstimates> estimates, Map<LocalDate, List<Double>> ratiosWithoutOutliers) {
        TreeMap<ClientsKey, ClientsRanges> ranges = new TreeMap<ClientsKey, ClientsRanges>();
        for (Map.Entry<ClientsKey, ClientsEstimates> estimatesEntry : estimates.entrySet()) {
            ClientsEstimates referenceEstimate;
            LocalDate date = estimatesEntry.getKey().date;
            if (!estimatesEntry.getKey().nodeIsRelay || "".equals(estimatesEntry.getKey().country) || "??".equals(estimatesEntry.getKey().country) || !ratiosWithoutOutliers.containsKey(date) || null == (referenceEstimate = (ClientsEstimates)estimates.get(new ClientsKey(date.minusDays(7L), true, estimatesEntry.getKey().country))) || referenceEstimate.clients == 0) continue;
            double[] values = ratiosWithoutOutliers.get(date).stream().mapToDouble(Double::doubleValue).toArray();
            double mean = new Mean().evaluate(values);
            double std = new StandardDeviation(false).evaluate(values);
            NormalDistribution normalDistribution = new NormalDistribution(mean, std);
            PoissonDistribution poissonDistribution = new PoissonDistribution(referenceEstimate.clients);
            int lower = Math.max(0, (int)(normalDistribution.inverseCumulativeProbability(1.0E-4) * (double)poissonDistribution.inverseCumulativeProbability(1.0E-4)));
            int upper = (int)(normalDistribution.inverseCumulativeProbability(0.9999) * (double)poissonDistribution.inverseCumulativeProbability(0.9999));
            ranges.put(estimatesEntry.getKey(), new ClientsRanges(lower, upper));
        }
        return ranges;
    }

    private static void writeOutputFile(SortedMap<ClientsKey, ClientsEstimates> estimates, SortedMap<ClientsKey, ClientsRanges> ranges) throws IOException {
        try (BufferedWriter bw = new BufferedWriter(new FileWriter(OUTPUT_PATH.toFile()));){
            bw.write("date,node,country,transport,version,lower,upper,clients,frac\n");
            for (Map.Entry<ClientsKey, ClientsEstimates> e : estimates.entrySet()) {
                String rangesString = ",";
                if (ranges.containsKey(e.getKey())) {
                    rangesString = ((ClientsRanges)ranges.get(e.getKey())).toString();
                }
                bw.write(String.format("%s,%s,%s%n", e.getKey().toString(), rangesString, e.getValue().toString()));
            }
        }
    }

    private static class ClientsRanges {
        private int lower;
        private int upper;

        ClientsRanges(int lower, int upper) {
            this.lower = lower;
            this.upper = upper;
        }

        public String toString() {
            return String.format("%d,%d", this.lower, this.upper);
        }
    }

    private static class ClientsEstimates {
        private int clients;
        private int frac;

        ClientsEstimates(int clients, int frac) {
            this.clients = clients;
            this.frac = frac;
        }

        public String toString() {
            return String.format("%d,%d", this.clients, this.frac);
        }
    }

    private static class ClientsKey
    implements Comparable<ClientsKey> {
        private LocalDate date;
        private boolean nodeIsRelay;
        private String country;
        private String transport = "";
        private String version = "";

        ClientsKey(LocalDate date, boolean nodeIsRelay, String country) {
            this.date = date;
            this.nodeIsRelay = nodeIsRelay;
            this.country = country;
        }

        ClientsKey(LocalDate date, boolean nodeIsRelay, String country, String transport, String version) {
            this(date, nodeIsRelay, country);
            this.transport = transport;
            this.version = version;
        }

        @Override
        public int compareTo(ClientsKey other) {
            if (!this.date.equals(other.date)) {
                return this.date.compareTo(other.date);
            }
            if (!this.nodeIsRelay && other.nodeIsRelay) {
                return -1;
            }
            if (this.nodeIsRelay && !other.nodeIsRelay) {
                return 1;
            }
            if (!this.country.equals(other.country)) {
                return this.country.compareTo(other.country);
            }
            if (!this.transport.equals(other.transport)) {
                return this.transport.compareTo(other.transport);
            }
            if (!this.version.equals(other.version)) {
                return this.version.compareTo(other.version);
            }
            return 0;
        }

        public boolean equals(Object otherObject) {
            if (!(otherObject instanceof ClientsKey)) {
                return false;
            }
            ClientsKey other = (ClientsKey)otherObject;
            return this.date.equals(other.date) && this.nodeIsRelay == other.nodeIsRelay && this.country.equals(other.country) && this.transport.equals(other.transport) && this.version.equals(other.version);
        }

        public int hashCode() {
            return 3 * this.date.hashCode() + (this.nodeIsRelay ? 5 : 0) + 7 * this.country.hashCode() + 11 * this.transport.hashCode() + 13 * this.version.hashCode();
        }

        public String toString() {
            return String.format("%s,%s,%s,%s,%s", this.date.toString(), this.nodeIsRelay ? "relay" : "bridge", this.country, this.transport, this.version);
        }
    }
}

