/*
 * Decompiled with CFR 0.152.
 */
package org.torproject.metrics.stats.connbidirect;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.StringReader;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TimeZone;
import java.util.TreeMap;
import java.util.TreeSet;
import org.apache.commons.math3.stat.descriptive.rank.Percentile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.torproject.descriptor.Descriptor;
import org.torproject.descriptor.DescriptorReader;
import org.torproject.descriptor.DescriptorSourceFactory;
import org.torproject.descriptor.ExtraInfoDescriptor;

public class Main {
    private static Logger log = LoggerFactory.getLogger(Main.class);
    static final long ONE_DAY_IN_MILLIS = 86400000L;
    private static final File baseDir = new File(org.torproject.metrics.stats.main.Main.modulesDir, "connbidirect");
    private static final String AGGREGATE_STATS_HEADER = "date,direction,quantile,fraction";

    public static void main(String[] args) throws IOException {
        SortedMap<String, Short> aggregateStats;
        File parseHistoryFile = new File(baseDir, "stats/parse-history");
        File aggregateStatsFile = new File(baseDir, "stats/connbidirect2.csv");
        File[] descriptorsDirectories = new File[]{new File(org.torproject.metrics.stats.main.Main.descriptorsDir, "archive/relay-descriptors/extra-infos"), new File(org.torproject.metrics.stats.main.Main.descriptorsDir, "recent/relay-descriptors/extra-infos")};
        SortedMap<String, Long> parseHistory = Main.parseParseHistory(Main.readStringFromFile(parseHistoryFile));
        if (parseHistory == null) {
            log.warn("Could not parse {}. Proceeding without parse history.", (Object)parseHistoryFile.getAbsolutePath());
        }
        if ((aggregateStats = Main.parseAggregateStats(Main.readStringFromFile(aggregateStatsFile))) == null) {
            log.warn("Could not parse previously aggregated statistics.  Not proceeding, because we would otherwise lose previously aggregated values for which we don't have raw statistics anymore.");
            return;
        }
        TreeSet<RawStat> newRawStats = new TreeSet<RawStat>();
        if ((parseHistory = Main.addRawStatsFromDescriptors(newRawStats, descriptorsDirectories, parseHistory)) == null) {
            log.warn("Could not parse raw statistics from descriptors.  Not proceeding, because we would otherwise leave out those descriptors in future runs.");
            return;
        }
        File rawStatsFile = new File(baseDir, "stats/raw-stats");
        SortedSet<RawStat> rawStats = Main.parseRawStats(Main.readStringFromFile(rawStatsFile));
        if (rawStats == null) {
            log.warn("Could not parse previously parsed raw statistics.  Not proceeding, because we might otherwise leave out previously parsed statistics in the aggregates.");
            return;
        }
        SortedSet<Long> conflictingDates = Main.mergeRawStats(rawStats, newRawStats);
        if (!conflictingDates.isEmpty()) {
            StringBuilder sb = new StringBuilder("Could not update aggregate statistics, because we already aggregated statistics for at least one contained date and discarded the underlying raw statistics.  Not proceeding.  To fix this, you'll have to re-import statistics for the following dates:");
            SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
            dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
            Iterator iterator = conflictingDates.iterator();
            while (iterator.hasNext()) {
                long conflictingDate = (Long)iterator.next();
                sb.append("\n ").append(dateFormat.format(conflictingDate * 86400000L));
            }
            log.warn(sb.toString());
            return;
        }
        Main.updateAggregateStats(aggregateStats, rawStats);
        Main.writeStringToFile(aggregateStatsFile, Main.formatAggregateStats(aggregateStats));
        Main.writeStringToFile(rawStatsFile, Main.formatRawStats(rawStats));
        Main.writeStringToFile(parseHistoryFile, Main.formatParseHistory(parseHistory));
    }

    private static String readStringFromFile(File file) throws IOException {
        StringBuilder sb = new StringBuilder();
        if (file.exists()) {
            String line;
            BufferedReader br = new BufferedReader(new FileReader(file));
            while ((line = br.readLine()) != null) {
                sb.append(line).append("\n");
            }
            br.close();
        }
        return sb.toString();
    }

    private static void writeStringToFile(File file, String string) throws IOException {
        file.getParentFile().mkdirs();
        File tempFile = new File(file.getParentFile(), file.getName() + ".tmp");
        BufferedWriter bw = new BufferedWriter(new FileWriter(tempFile));
        bw.write(string);
        bw.close();
        tempFile.renameTo(file);
    }

    static String formatParseHistory(SortedMap<String, Long> parseHistory) {
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String, Long> e : parseHistory.entrySet()) {
            sb.append(e.getKey()).append(",").append(e.getValue()).append("\n");
        }
        return sb.toString();
    }

    static SortedMap<String, Long> parseParseHistory(String formattedParseHistory) {
        TreeMap<String, Long> parsedParseHistory = new TreeMap<String, Long>();
        LineNumberReader lnr = new LineNumberReader(new StringReader(formattedParseHistory));
        String line = "";
        try {
            while ((line = lnr.readLine()) != null) {
                String[] parts = line.split(",");
                if (parts.length < 2) {
                    log.warn("Invalid line {} in parse history: '{}'.", (Object)lnr.getLineNumber(), (Object)line);
                    return null;
                }
                parsedParseHistory.put(parts[0], Long.parseLong(parts[1]));
            }
        }
        catch (IOException e) {
            log.warn("Unexpected I/O exception while reading line {} from parse history.", (Object)lnr.getLineNumber(), (Object)e);
            return null;
        }
        catch (NumberFormatException e) {
            log.warn("Invalid line {} in parse history: '{}'.", lnr.getLineNumber(), line, e);
            return null;
        }
        return parsedParseHistory;
    }

    static String formatAggregateStats(SortedMap<String, Short> aggregateStats) {
        StringBuilder sb = new StringBuilder();
        sb.append("date,direction,quantile,fraction\n");
        for (Map.Entry<String, Short> e : aggregateStats.entrySet()) {
            sb.append(e.getKey()).append(",").append(e.getValue()).append("\n");
        }
        return sb.toString();
    }

    static SortedMap<String, Short> parseAggregateStats(String formattedAggregatedStats) {
        TreeMap<String, Short> parsedAggregateStats = new TreeMap<String, Short>();
        if (formattedAggregatedStats.length() < 1) {
            return parsedAggregateStats;
        }
        LineNumberReader lnr = new LineNumberReader(new StringReader(formattedAggregatedStats));
        String line = "";
        try {
            if (!AGGREGATE_STATS_HEADER.equals(lnr.readLine())) {
                log.warn("First line of aggregate statistics does not contain the header line. Is this the correct file?");
                return null;
            }
            while ((line = lnr.readLine()) != null) {
                String[] parts = line.split(",");
                if (parts.length != 4) {
                    log.warn("Invalid line {} in aggregate statistics: '{}'.", (Object)lnr.getLineNumber(), (Object)line);
                    return null;
                }
                parsedAggregateStats.put(parts[0] + "," + parts[1] + "," + parts[2], Short.parseShort(parts[3]));
            }
        }
        catch (IOException e) {
            log.warn("Unexpected I/O exception while reading line {} from aggregate statistics.", (Object)lnr.getLineNumber(), (Object)e);
            return null;
        }
        catch (NumberFormatException e) {
            log.warn("Invalid line {} in aggregate statistics: '{}'.", lnr.getLineNumber(), line, e);
            return null;
        }
        return parsedAggregateStats;
    }

    static String formatRawStats(SortedSet<RawStat> rawStats) {
        StringBuilder sb = new StringBuilder();
        for (RawStat rawStat : rawStats) {
            sb.append(rawStat.toString()).append("\n");
        }
        return sb.toString();
    }

    static SortedSet<RawStat> parseRawStats(String formattedRawStats) {
        TreeSet<RawStat> parsedRawStats = new TreeSet<RawStat>();
        LineNumberReader lnr = new LineNumberReader(new StringReader(formattedRawStats));
        String line = "";
        try {
            while ((line = lnr.readLine()) != null) {
                RawStat rawStat = RawStat.fromString(line);
                if (rawStat == null) {
                    log.warn("Invalid line {} in raw statistics: '{}'.", (Object)lnr.getLineNumber(), (Object)line);
                    return null;
                }
                parsedRawStats.add(rawStat);
            }
        }
        catch (IOException e) {
            log.warn("Unexpected I/O exception while reading line {} from raw statistics.", (Object)lnr.getLineNumber(), (Object)e);
            return null;
        }
        catch (NumberFormatException e) {
            log.warn("Invalid line {} in raw statistics: '{}'.", lnr.getLineNumber(), line, e);
            return null;
        }
        return parsedRawStats;
    }

    private static SortedMap<String, Long> addRawStatsFromDescriptors(SortedSet<RawStat> rawStats, File[] descriptorsDirectories, SortedMap<String, Long> parseHistory) {
        DescriptorReader descriptorReader = DescriptorSourceFactory.createDescriptorReader();
        descriptorReader.setExcludedFiles(parseHistory);
        for (Descriptor descriptor : descriptorReader.readDescriptors(descriptorsDirectories)) {
            RawStat rawStat;
            if (!(descriptor instanceof ExtraInfoDescriptor) || (rawStat = Main.parseRawStatFromDescriptor((ExtraInfoDescriptor)descriptor)) == null) continue;
            rawStats.add(rawStat);
        }
        parseHistory.clear();
        parseHistory.putAll(descriptorReader.getExcludedFiles());
        parseHistory.putAll(descriptorReader.getParsedFiles());
        return parseHistory;
    }

    private static RawStat parseRawStatFromDescriptor(ExtraInfoDescriptor extraInfo) {
        if (extraInfo.getConnBiDirectStatsEndMillis() <= 0L) {
            return null;
        }
        int below = extraInfo.getConnBiDirectBelow();
        int read = extraInfo.getConnBiDirectRead();
        int write = extraInfo.getConnBiDirectWrite();
        int both = extraInfo.getConnBiDirectBoth();
        if (below < 0 || read < 0 || write < 0 || both < 0) {
            log.debug("Could not parse incomplete conn-bi-direct statistics. Skipping descriptor.");
            return null;
        }
        long statsEndMillis = extraInfo.getConnBiDirectStatsEndMillis();
        String fingerprint = extraInfo.getFingerprint();
        return Main.parseRawStatFromDescriptorContents(statsEndMillis, fingerprint, below, read, write, both);
    }

    static RawStat parseRawStatFromDescriptorContents(long statsEndMillis, String fingerprint, int below, int read, int write, int both) {
        int total = read + write + both;
        if (below < 0 || read < 0 || write < 0 || both < 0 || total <= 0) {
            return null;
        }
        long dateDays = statsEndMillis / 86400000L;
        short fractionRead = (short)(read * 100 / total);
        short fractionWrite = (short)(write * 100 / total);
        short fractionBoth = (short)(both * 100 / total);
        return new RawStat(dateDays, fingerprint, fractionRead, fractionWrite, fractionBoth);
    }

    static SortedSet<Long> mergeRawStats(SortedSet<RawStat> rawStats, SortedSet<RawStat> newRawStats) {
        rawStats.addAll(newRawStats);
        TreeSet<Long> discardedRawStats = new TreeSet<Long>();
        TreeSet<Long> availableRawStats = new TreeSet<Long>();
        for (RawStat rawStat : rawStats) {
            if (rawStat.fingerprint != null) {
                availableRawStats.add(rawStat.dateDays);
                continue;
            }
            discardedRawStats.add(rawStat.dateDays);
        }
        discardedRawStats.retainAll(availableRawStats);
        return discardedRawStats;
    }

    static void updateAggregateStats(SortedMap<String, Short> aggregateStats, SortedSet<RawStat> rawStats) {
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
        dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        String yesterday = dateFormat.format(System.currentTimeMillis() - 86400000L);
        TreeMap fractionsByDateAndDirection = new TreeMap();
        String[] directions = new String[]{"read", "write", "both"};
        for (RawStat rawStat : rawStats) {
            String date;
            if (rawStat.fingerprint == null || (date = dateFormat.format(rawStat.dateDays * 86400000L)).compareTo(yesterday) >= 0) continue;
            short[] fractions = new short[]{rawStat.fractionRead, rawStat.fractionWrite, rawStat.fractionBoth};
            for (int i = 0; i < directions.length; ++i) {
                String dateAndDirection = date + "," + directions[i];
                fractionsByDateAndDirection.putIfAbsent(dateAndDirection, new ArrayList());
                ((List)fractionsByDateAndDirection.get(dateAndDirection)).add(fractions[i]);
            }
        }
        String[] quantiles = new String[]{"0.25", "0.5", "0.75"};
        double[] centiles = new double[]{25.0, 50.0, 75.0};
        for (Map.Entry e : fractionsByDateAndDirection.entrySet()) {
            String dateAndDirection = (String)e.getKey();
            List fractions = (List)e.getValue();
            SortedMap<Double, Short> computedPercentiles = Main.computePercentiles(fractions, centiles);
            for (int i = 0; i < quantiles.length; ++i) {
                String dateDirectionAndQuantile = dateAndDirection + "," + quantiles[i];
                short fraction = (Short)computedPercentiles.get(centiles[i]);
                aggregateStats.put(dateDirectionAndQuantile, fraction);
            }
        }
    }

    static SortedMap<Double, Short> computePercentiles(List<Short> valueList, double ... percentiles) {
        TreeMap<Double, Short> computedPercentiles = new TreeMap<Double, Short>();
        double[] valueArray = new double[valueList.size()];
        for (int i = 0; i < valueList.size(); ++i) {
            valueArray[i] = valueList.get(i).doubleValue();
        }
        Percentile percentile = new Percentile().withEstimationType(Percentile.EstimationType.R_7);
        percentile.setData(valueArray);
        for (double p : percentiles) {
            computedPercentiles.put(p, (short)Math.floor(percentile.evaluate(p)));
        }
        return computedPercentiles;
    }

    static class RawStat
    implements Comparable<RawStat> {
        long dateDays;
        String fingerprint;
        short fractionRead;
        short fractionWrite;
        short fractionBoth;

        RawStat(long dateDays, String fingerprint, short fractionRead, short fractionWrite, short fractionBoth) {
            this.dateDays = dateDays;
            this.fingerprint = fingerprint;
            this.fractionRead = fractionRead;
            this.fractionWrite = fractionWrite;
            this.fractionBoth = fractionBoth;
        }

        static RawStat fromString(String string) {
            try {
                String[] parts = string.split(",");
                if (parts.length == 5) {
                    long dateDays = Long.parseLong(parts[0]);
                    String fingerprint = parts[1];
                    short fractionRead = Short.parseShort(parts[2]);
                    short fractionWrite = Short.parseShort(parts[3]);
                    short fractionBoth = Short.parseShort(parts[4]);
                    return new RawStat(dateDays, fingerprint, fractionRead, fractionWrite, fractionBoth);
                }
                log.warn("Could not deserialize raw statistic from string '{}'.", (Object)string);
                return null;
            }
            catch (NumberFormatException e) {
                log.warn("Could not deserialize raw statistic from string '{}'.", (Object)string, (Object)e);
                return null;
            }
        }

        public String toString() {
            if (this.fingerprint == null) {
                return String.valueOf(this.dateDays);
            }
            return String.format("%d,%s,%d,%d,%d", this.dateDays, this.fingerprint, this.fractionRead, this.fractionWrite, this.fractionBoth);
        }

        @Override
        public int compareTo(RawStat other) {
            if (this.dateDays != other.dateDays) {
                return this.dateDays < other.dateDays ? -1 : 1;
            }
            if (this.fingerprint != null && other.fingerprint != null) {
                return this.fingerprint.compareTo(other.fingerprint);
            }
            if (this.fingerprint != null) {
                return -1;
            }
            if (other.fingerprint != null) {
                return 1;
            }
            return 0;
        }

        public boolean equals(Object otherObject) {
            if (!(otherObject instanceof RawStat)) {
                return false;
            }
            RawStat other = (RawStat)otherObject;
            return this.dateDays == other.dateDays && this.fingerprint.equals(other.fingerprint);
        }
    }
}

