/*
 * Decompiled with CFR 0.152.
 */
package org.torproject.metrics.exonerator;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Calendar;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TimeZone;
import java.util.TreeMap;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.torproject.descriptor.Descriptor;
import org.torproject.descriptor.DescriptorCollector;
import org.torproject.descriptor.DescriptorReader;
import org.torproject.descriptor.DescriptorSourceFactory;
import org.torproject.descriptor.ExitList;
import org.torproject.descriptor.NetworkStatusEntry;
import org.torproject.descriptor.RelayNetworkStatusConsensus;

public class ExoneraTorDatabaseImporter {
    private static Logger logger = LoggerFactory.getLogger(ExoneraTorDatabaseImporter.class);
    private static String jdbcString;
    private static String importDirString;
    private static Connection connection;
    private static CallableStatement insertStatusentryStatement;
    private static CallableStatement insertExitlistentryStatement;
    private static SortedMap<String, Long> lastImportHistory;
    private static SortedMap<String, Long> nextImportHistory;
    private static Calendar calendarUTC;

    public static void main(String[] args) {
        ExoneraTorDatabaseImporter.readConfiguration();
        ExoneraTorDatabaseImporter.openDatabaseConnection();
        ExoneraTorDatabaseImporter.prepareDatabaseStatements();
        ExoneraTorDatabaseImporter.createLockFile();
        ExoneraTorDatabaseImporter.fetchDescriptors();
        ExoneraTorDatabaseImporter.readImportHistoryToMemory();
        ExoneraTorDatabaseImporter.parseDescriptors();
        ExoneraTorDatabaseImporter.writeImportHistoryToDisk();
        ExoneraTorDatabaseImporter.closeDatabaseConnection();
        ExoneraTorDatabaseImporter.deleteLockFile();
    }

    private static void readConfiguration() {
        File configFile = new File("config");
        if (!configFile.exists()) {
            logger.error("Could not find config file.  Exiting.");
            System.exit(1);
        }
        try {
            String line;
            BufferedReader br = new BufferedReader(new FileReader(configFile));
            while ((line = br.readLine()) != null) {
                if (line.startsWith("#") || line.length() < 1) continue;
                if (line.startsWith("ExoneraTorDatabaseJdbc")) {
                    jdbcString = line.split(" ")[1];
                    continue;
                }
                if (!line.startsWith("ExoneraTorImportDirectory")) continue;
                importDirString = line.split(" ")[1];
            }
            br.close();
        }
        catch (IOException e) {
            logger.error("Could not parse config file.  Exiting.", e);
            System.exit(1);
        }
    }

    private static void openDatabaseConnection() {
        try {
            connection = DriverManager.getConnection(jdbcString);
        }
        catch (SQLException e) {
            logger.error("Could not connect to database.  Exiting.", e);
            System.exit(1);
        }
    }

    private static void prepareDatabaseStatements() {
        try {
            insertStatusentryStatement = connection.prepareCall("{call insert_statusentry_oraddress(?, ?, ?, ?, ?, ?)}");
            insertExitlistentryStatement = connection.prepareCall("{call insert_exitlistentry_exitaddress(?, ?, ?, ?)}");
        }
        catch (SQLException e) {
            logger.warn("Could not prepare callable statements to import data into the database.  Exiting.", e);
            System.exit(1);
        }
    }

    private static void createLockFile() {
        File lockFile = new File("exonerator-lock");
        try {
            if (lockFile.exists()) {
                BufferedReader br = new BufferedReader(new FileReader(lockFile));
                long runStarted = Long.parseLong(br.readLine());
                br.close();
                if (System.currentTimeMillis() - runStarted < 21600000L) {
                    logger.warn("File 'exonerator-lock' is less than 6 hours old.  Exiting.");
                    System.exit(1);
                } else {
                    logger.warn("File 'exonerator-lock' is at least 6 hours old.  Overwriting and executing anyway.");
                }
            }
            BufferedWriter bw = new BufferedWriter(new FileWriter("exonerator-lock"));
            bw.append(String.valueOf(System.currentTimeMillis())).append("\n");
            bw.close();
        }
        catch (IOException e) {
            logger.warn("Could not create 'exonerator-lock' file.  Exiting.");
            System.exit(1);
        }
    }

    private static void fetchDescriptors() {
        DescriptorCollector collector = DescriptorSourceFactory.createDescriptorCollector();
        collector.collectDescriptors("https://collector.torproject.org", new String[]{"/recent/relay-descriptors/consensuses/", "/recent/exit-lists/"}, 0L, new File(importDirString), true);
    }

    private static void readImportHistoryToMemory() {
        File parseHistoryFile = new File("stats", "exonerator-import-history");
        if (parseHistoryFile.exists()) {
            try {
                String line;
                BufferedReader br = new BufferedReader(new FileReader(parseHistoryFile));
                int lineNumber = 0;
                while ((line = br.readLine()) != null) {
                    ++lineNumber;
                    String[] parts = line.split(",");
                    if (parts.length != 2) {
                        logger.warn("File 'stats/exonerator-import-history' contains a corrupt entry in line {}.  Ignoring parse history file entirely.", (Object)lineNumber);
                        lastImportHistory.clear();
                        br.close();
                        return;
                    }
                    long lastModified = Long.parseLong(parts[0]);
                    String filename = parts[1];
                    lastImportHistory.put(filename, lastModified);
                }
                br.close();
            }
            catch (IOException e) {
                logger.warn("Could not read import history.  Ignoring.", e);
                lastImportHistory.clear();
            }
        }
    }

    private static void parseDescriptors() {
        DescriptorReader descriptorReader = DescriptorSourceFactory.createDescriptorReader();
        descriptorReader.setMaxDescriptorsInQueue(20);
        descriptorReader.setExcludedFiles(lastImportHistory);
        for (Descriptor descriptor : descriptorReader.readDescriptors(new File(importDirString))) {
            if (descriptor instanceof RelayNetworkStatusConsensus) {
                ExoneraTorDatabaseImporter.parseConsensus((RelayNetworkStatusConsensus)descriptor);
                continue;
            }
            if (!(descriptor instanceof ExitList)) continue;
            ExoneraTorDatabaseImporter.parseExitList((ExitList)descriptor);
        }
        nextImportHistory.putAll(descriptorReader.getExcludedFiles());
        nextImportHistory.putAll(descriptorReader.getParsedFiles());
    }

    private static void parseConsensus(RelayNetworkStatusConsensus consensus) {
        long validAfterMillis = consensus.getValidAfterMillis();
        for (NetworkStatusEntry entry : consensus.getStatusEntries().values()) {
            if (!entry.getFlags().contains("Running")) continue;
            String fingerprintBase64 = null;
            try {
                fingerprintBase64 = Base64.encodeBase64String(Hex.decodeHex(entry.getFingerprint().toCharArray())).replaceAll("=", "");
            }
            catch (DecoderException e) {
                logger.warn("Unable to decode hex fingerprint {} to convert it back to base64. Aborting import.", (Object)entry.getFingerprint(), (Object)e);
                System.exit(1);
            }
            String nickname = entry.getNickname();
            Boolean exit = null;
            if (null != entry.getDefaultPolicy() && null != entry.getPortList()) {
                exit = "accept".equals(entry.getDefaultPolicy()) || !"1-65535".equals(entry.getPortList());
            }
            HashSet<String> orAddresses = new HashSet<String>();
            orAddresses.add(entry.getAddress());
            for (String orAddressAndPort : entry.getOrAddresses()) {
                orAddresses.add(orAddressAndPort.substring(0, orAddressAndPort.lastIndexOf(":")));
            }
            ExoneraTorDatabaseImporter.importStatusentry(validAfterMillis, fingerprintBase64, nickname, exit, orAddresses);
        }
    }

    private static void importStatusentry(long validAfterMillis, String fingerprintBase64, String nickname, Boolean exit, Set<String> orAddresses) {
        try {
            for (String orAddress : orAddresses) {
                insertStatusentryStatement.clearParameters();
                insertStatusentryStatement.setTimestamp(1, new Timestamp(validAfterMillis), calendarUTC);
                insertStatusentryStatement.setString(2, fingerprintBase64);
                if (!orAddress.contains(":")) {
                    insertStatusentryStatement.setString(3, orAddress);
                    String[] addressParts = orAddress.split("\\.");
                    byte[] address24Bytes = new byte[]{(byte)Integer.parseInt(addressParts[0]), (byte)Integer.parseInt(addressParts[1]), (byte)Integer.parseInt(addressParts[2])};
                    String orAddress24 = Hex.encodeHexString(address24Bytes);
                    insertStatusentryStatement.setString(4, orAddress24);
                } else {
                    String[] parts;
                    StringBuilder addressHex = new StringBuilder();
                    int start = orAddress.startsWith("[::") ? 2 : 1;
                    int end = orAddress.length() - (orAddress.endsWith("::]") ? 2 : 1);
                    for (String part : parts = orAddress.substring(start, end).split(":", -1)) {
                        if (part.length() == 0) {
                            addressHex.append("x");
                            continue;
                        }
                        if (part.length() <= 4) {
                            addressHex.append(String.format("%4s", part));
                            continue;
                        }
                        addressHex = null;
                        break;
                    }
                    String orAddress24 = null;
                    if (addressHex != null) {
                        String addressHexString = addressHex.toString();
                        if (!(addressHexString = addressHexString.replaceFirst("x", String.format("%" + (33 - addressHexString.length()) + "s", "0"))).contains("x") && addressHexString.length() == 32) {
                            orAddress24 = addressHexString.replaceAll(" ", "0").toLowerCase().substring(0, 6);
                        }
                    }
                    if (orAddress24 != null) {
                        insertStatusentryStatement.setString(3, orAddress.replaceAll("[\\[\\]]", ""));
                        insertStatusentryStatement.setString(4, orAddress24);
                    } else {
                        logger.error("Could not import status entry with IPv6 address '{}'.  Exiting.", (Object)orAddress);
                        System.exit(1);
                    }
                }
                insertStatusentryStatement.setString(5, nickname);
                insertStatusentryStatement.setBoolean(6, (boolean)exit);
                insertStatusentryStatement.execute();
            }
        }
        catch (SQLException e) {
            logger.error("Could not import status entry.  Exiting.", e);
            System.exit(1);
        }
    }

    private static void parseExitList(ExitList exitList) {
        for (ExitList.Entry entry : exitList.getEntries()) {
            for (Map.Entry<String, Long> e : entry.getExitAddresses().entrySet()) {
                String fingerprintBase64 = null;
                try {
                    fingerprintBase64 = Base64.encodeBase64String(Hex.decodeHex(entry.getFingerprint().toCharArray())).replaceAll("=", "");
                }
                catch (DecoderException ex) {
                    logger.warn("Unable to decode hex fingerprint {} to convert to base64. Aborting import.", (Object)entry.getFingerprint(), (Object)ex);
                    System.exit(1);
                }
                String exitAddress = e.getKey();
                String[] exitAddressParts = exitAddress.split("\\.");
                byte[] exitAddress24Bytes = new byte[]{(byte)Integer.parseInt(exitAddressParts[0]), (byte)Integer.parseInt(exitAddressParts[1]), (byte)Integer.parseInt(exitAddressParts[2])};
                String exitAddress24 = Hex.encodeHexString(exitAddress24Bytes);
                long scannedMillis = e.getValue();
                ExoneraTorDatabaseImporter.importExitlistentry(fingerprintBase64, exitAddress24, exitAddress, scannedMillis);
            }
        }
    }

    private static void importExitlistentry(String fingerprintBase64, String exitAddress24, String exitAddress, long scannedMillis) {
        try {
            insertExitlistentryStatement.clearParameters();
            insertExitlistentryStatement.setString(1, fingerprintBase64);
            insertExitlistentryStatement.setString(2, exitAddress);
            insertExitlistentryStatement.setString(3, exitAddress24);
            insertExitlistentryStatement.setTimestamp(4, new Timestamp(scannedMillis), calendarUTC);
            insertExitlistentryStatement.execute();
        }
        catch (SQLException e) {
            logger.error("Could not import exit list entry.  Exiting.", e);
            System.exit(1);
        }
    }

    private static void writeImportHistoryToDisk() {
        File parseHistoryFile = new File("stats/exonerator-import-history");
        parseHistoryFile.getParentFile().mkdirs();
        try {
            BufferedWriter bw = new BufferedWriter(new FileWriter(parseHistoryFile));
            for (Map.Entry<String, Long> historyEntry : nextImportHistory.entrySet()) {
                bw.write(String.valueOf(historyEntry.getValue()) + "," + historyEntry.getKey() + "\n");
            }
            bw.close();
        }
        catch (IOException e) {
            logger.warn("File 'stats/exonerator-import-history' could not be written.  Ignoring.", e);
        }
    }

    private static void closeDatabaseConnection() {
        try {
            connection.close();
        }
        catch (SQLException e) {
            logger.warn("Could not close database connection. Ignoring.", e);
        }
    }

    private static void deleteLockFile() {
        new File("exonerator-lock").delete();
    }

    static {
        lastImportHistory = new TreeMap<String, Long>();
        nextImportHistory = new TreeMap<String, Long>();
        calendarUTC = Calendar.getInstance(TimeZone.getTimeZone("UTC"));
    }
}

