/*
 * Decompiled with CFR 0.152.
 */
package org.torproject.metrics.stats.webstats;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.sql.Connection;
import java.sql.Date;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Locale;
import java.util.Map;
import java.util.SortedSet;
import java.util.TimeZone;
import java.util.TreeSet;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.torproject.descriptor.Descriptor;
import org.torproject.descriptor.DescriptorParseException;
import org.torproject.descriptor.DescriptorSourceFactory;
import org.torproject.descriptor.WebServerAccessLog;

public class Main {
    private static Logger log = LoggerFactory.getLogger(Main.class);
    private static final String LOG_DATE = "log_date";
    private static final String REQUEST_TYPE = "request_type";
    private static final String PLATFORM = "platform";
    private static final String CHANNEL = "channel";
    private static final String LOCALE = "locale";
    private static final String INCREMENTAL = "incremental";
    private static final String COUNT = "count";
    private static final String ALL_COLUMNS = "log_date,request_type,platform,channel,locale,incremental,count";

    public static void main(String[] args) throws Exception {
        log.info("Starting webstats module.");
        String dbUrlString = "jdbc:postgresql:webstats";
        Connection connection = Main.connectToDatabase(dbUrlString);
        SortedSet<String> skipFiles = Main.queryImportedFileNames(connection);
        Main.importLogFiles(connection, skipFiles, new File("../../shared/in/recent/webstats"), new File("../../shared/in/archive/webstats"));
        SortedSet<String> statistics = Main.queryWebstats(connection);
        Main.writeStatistics(Paths.get("stats", "webstats.csv"), statistics);
        Main.disconnectFromDatabase(connection);
        log.info("Terminated webstats module.");
    }

    private static Connection connectToDatabase(String jdbcString) throws SQLException {
        log.info("Connecting to database.");
        Connection connection = DriverManager.getConnection(jdbcString);
        connection.setAutoCommit(false);
        log.info("Successfully connected to database.");
        return connection;
    }

    static SortedSet<String> queryImportedFileNames(Connection connection) throws SQLException {
        log.info("Querying previously imported log files.");
        TreeSet<String> importedLogFileUrls = new TreeSet<String>();
        Statement st = connection.createStatement();
        String queryString = "SELECT server, site, log_date FROM files";
        DateTimeFormatter dateFormat = DateTimeFormatter.ofPattern("yyyyMMdd");
        try (ResultSet rs = st.executeQuery(queryString);){
            while (rs.next()) {
                importedLogFileUrls.add(String.format("%s_%s_access.log_%s.xz", rs.getString(1), rs.getString(2), rs.getDate(3).toLocalDate().format(dateFormat)));
            }
        }
        log.info("Found {} previously imported log files.", (Object)importedLogFileUrls.size());
        return importedLogFileUrls;
    }

    static void importLogFiles(Connection connection, SortedSet<String> skipFiles, File ... inDirectories) {
        for (Descriptor descriptor : DescriptorSourceFactory.createDescriptorReader().readDescriptors(inDirectories)) {
            WebServerAccessLog logFile;
            if (!(descriptor instanceof WebServerAccessLog) || skipFiles.contains((logFile = (WebServerAccessLog)descriptor).getDescriptorFile().getName())) continue;
            try {
                Map parsedLogLines = ((Stream)logFile.logLines().parallel()).map(line -> line).collect(Collectors.groupingByConcurrent(line -> String.format("%s %s %d", line.getMethod().name(), Main.truncateString(line.getRequest(), 2048), line.getResponse()), Collectors.counting()));
                Main.importLogLines(connection, logFile.getDescriptorFile().getName(), logFile.getPhysicalHost(), logFile.getVirtualHost(), logFile.getLogDate(), parsedLogLines);
            }
            catch (DescriptorParseException exc) {
                log.warn("Cannot parse log file with file name {}.  Retrying in the next run.", (Object)logFile.getDescriptorFile().getName(), (Object)exc);
            }
            catch (SQLException exc) {
                log.warn("Cannot import log file with file name {} into the database. Rolling back and retrying in the next run.", (Object)logFile.getDescriptorFile().getName(), (Object)exc);
                try {
                    connection.rollback();
                }
                catch (SQLException sQLException) {}
            }
        }
    }

    private static void importLogLines(Connection connection, String urlString, String server, String site, LocalDate logDate, Map<String, Long> parsedLogLines) throws SQLException {
        PreparedStatement psFiles = connection.prepareStatement("INSERT INTO files (url, server, site, log_date) VALUES (?, ?, ?, ?)", 1);
        PreparedStatement psResourcesSelect = connection.prepareStatement("SELECT resource_id FROM resources WHERE resource_string = ?");
        PreparedStatement psResourcesInsert = connection.prepareStatement("INSERT INTO resources (resource_string) VALUES (?)", 1);
        PreparedStatement psRequests = connection.prepareStatement("INSERT INTO requests (file_id, method, resource_id, response_code, count) VALUES (?, CAST(? AS method), ?, ?, ?)");
        int fileId = Main.insertFile(psFiles, urlString, server, site, logDate);
        if (fileId < 0) {
            log.debug("Skipping previously imported log file {}.", (Object)urlString);
            return;
        }
        for (Map.Entry<String, Long> requests : parsedLogLines.entrySet()) {
            String[] keyParts = requests.getKey().split(" ");
            String method = keyParts[0];
            String resource = keyParts[1];
            int responseCode = Integer.parseInt(keyParts[2]);
            long count = requests.getValue();
            int resourceId = Main.insertResource(psResourcesSelect, psResourcesInsert, resource);
            if (resourceId < 0) {
                log.error("Could not retrieve auto-generated key for new resources entry.");
                connection.rollback();
                return;
            }
            Main.insertRequest(psRequests, fileId, method, resourceId, responseCode, count);
        }
        connection.commit();
        log.debug("Finished importing log file with file name {} into database.", (Object)urlString);
    }

    private static int insertFile(PreparedStatement psFiles, String urlString, String server, String site, LocalDate logDate) throws SQLException {
        int fileId = -1;
        psFiles.clearParameters();
        psFiles.setString(1, Main.truncateString(urlString, 2048));
        psFiles.setString(2, Main.truncateString(server, 32));
        psFiles.setString(3, Main.truncateString(site, 128));
        psFiles.setDate(4, Date.valueOf(logDate));
        psFiles.execute();
        try (ResultSet rs = psFiles.getGeneratedKeys();){
            if (rs.next()) {
                fileId = rs.getInt(1);
            }
        }
        return fileId;
    }

    private static void insertRequest(PreparedStatement psRequests, int fileId, String method, int resourceId, int responseCode, long count) throws SQLException {
        psRequests.clearParameters();
        psRequests.setInt(1, fileId);
        psRequests.setString(2, method);
        psRequests.setInt(3, resourceId);
        psRequests.setInt(4, responseCode);
        psRequests.setLong(5, count);
        psRequests.execute();
    }

    private static int insertResource(PreparedStatement psResourcesSelect, PreparedStatement psResourcesInsert, String resource) throws SQLException {
        int resourceId = -1;
        String truncatedResource = Main.truncateString(resource, 2048);
        psResourcesSelect.clearParameters();
        psResourcesSelect.setString(1, truncatedResource);
        try (ResultSet rs = psResourcesSelect.executeQuery();){
            if (rs.next()) {
                resourceId = rs.getInt(1);
            }
        }
        if (resourceId < 0) {
            psResourcesInsert.clearParameters();
            psResourcesInsert.setString(1, truncatedResource);
            psResourcesInsert.execute();
            rs = psResourcesInsert.getGeneratedKeys();
            var6_6 = null;
            try {
                if (rs.next()) {
                    resourceId = rs.getInt(1);
                }
            }
            catch (Throwable throwable) {
                var6_6 = throwable;
                throw throwable;
            }
            finally {
                if (rs != null) {
                    if (var6_6 != null) {
                        try {
                            rs.close();
                        }
                        catch (Throwable throwable) {
                            var6_6.addSuppressed(throwable);
                        }
                    } else {
                        rs.close();
                    }
                }
            }
        }
        return resourceId;
    }

    private static String truncateString(String originalString, int truncateAfter) {
        if (originalString.length() > truncateAfter) {
            originalString = originalString.substring(0, truncateAfter);
        }
        return originalString;
    }

    static SortedSet<String> queryWebstats(Connection connection) throws SQLException {
        log.info("Querying statistics from database.");
        TreeSet<String> statistics = new TreeSet<String>();
        Statement st = connection.createStatement();
        String queryString = "SELECT log_date,request_type,platform,channel,locale,incremental,count FROM webstats";
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd", Locale.US);
        dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        Calendar calendar = Calendar.getInstance(TimeZone.getTimeZone("UTC"), Locale.US);
        try (ResultSet rs = st.executeQuery(queryString);){
            while (rs.next()) {
                statistics.add(String.format("%s,%s,%s,%s,%s,%s,%d", dateFormat.format(rs.getDate(LOG_DATE, calendar)), Main.emptyNull(rs.getString(REQUEST_TYPE)), Main.emptyNull(rs.getString(PLATFORM)), Main.emptyNull(rs.getString(CHANNEL)), Main.emptyNull(rs.getString(LOCALE)), Main.emptyNull(rs.getString(INCREMENTAL)), rs.getLong(COUNT)));
            }
        }
        return statistics;
    }

    private static String emptyNull(String text) {
        return null == text ? "" : text;
    }

    static void writeStatistics(Path webstatsPath, SortedSet<String> statistics) throws IOException {
        webstatsPath.toFile().getParentFile().mkdirs();
        ArrayList<String> lines = new ArrayList<String>();
        lines.add(ALL_COLUMNS);
        lines.addAll(statistics);
        log.info("Writing {} lines to {}.", (Object)lines.size(), (Object)webstatsPath.toFile().getAbsolutePath());
        Files.write(webstatsPath, lines, StandardCharsets.UTF_8, new OpenOption[0]);
    }

    private static void disconnectFromDatabase(Connection connection) throws SQLException {
        log.info("Disconnecting from database.");
        connection.close();
    }
}

