/*
 * Decompiled with CFR 0.152.
 */
package org.torproject.metrics.webstats;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.sql.Connection;
import java.sql.Date;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.SortedSet;
import java.util.TimeZone;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.compress.compressors.xz.XZCompressorInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Main {
    private static Logger log = LoggerFactory.getLogger(Main.class);
    static final Pattern URL_STRING_PATTERN = Pattern.compile(".*<a href=\"([^\"]+)\">.*");
    static final Pattern LOG_FILE_URL_PATTERN = Pattern.compile("^.*/([^/]+)/([^/]+)-access.log-(\\d{8}).xz$");
    private static DateFormat logDateFormat = new SimpleDateFormat("yyyyMMdd");
    static final Pattern LOG_LINE_PATTERN;
    private static final String LOG_DATE = "log_date";
    private static final String REQUEST_TYPE = "request_type";
    private static final String PLATFORM = "platform";
    private static final String CHANNEL = "channel";
    private static final String LOCALE = "locale";
    private static final String INCREMENTAL = "incremental";
    private static final String COUNT = "count";
    private static final String ALL_COLUMNS = "log_date,request_type,platform,channel,locale,incremental,count";

    public static void main(String[] args) throws Exception {
        log.info("Starting webstats module.");
        String dbUrlString = "jdbc:postgresql:webstats";
        Connection connection = Main.connectToDatabase(dbUrlString);
        SortedSet<String> previouslyImportedLogFileUrls = Main.queryImportedFiles(connection);
        String baseUrl = "https://webstats.torproject.org/out/";
        SortedSet<String> newLogFileUrls = Main.downloadDirectoryListings(baseUrl, previouslyImportedLogFileUrls);
        Main.importLogFiles(connection, newLogFileUrls);
        SortedSet<String> statistics = Main.queryWebstats(connection);
        Main.writeStatistics(Paths.get("stats", "webstats.csv"), statistics);
        Main.disconnectFromDatabase(connection);
        log.info("Terminated webstats module.");
    }

    private static Connection connectToDatabase(String jdbcString) throws SQLException {
        log.info("Connecting to database.");
        Connection connection = DriverManager.getConnection(jdbcString);
        connection.setAutoCommit(false);
        log.info("Successfully connected to database.");
        return connection;
    }

    static SortedSet<String> queryImportedFiles(Connection connection) throws SQLException {
        log.info("Querying URLs of previously imported log files.");
        TreeSet<String> importedLogFileUrls = new TreeSet<String>();
        Statement st = connection.createStatement();
        String queryString = "SELECT url FROM files";
        try (ResultSet rs = st.executeQuery(queryString);){
            while (rs.next()) {
                importedLogFileUrls.add(rs.getString(1));
            }
        }
        log.info("Found {} URLs of previously imported log files.", (Object)importedLogFileUrls.size());
        return importedLogFileUrls;
    }

    static SortedSet<String> downloadDirectoryListings(String baseUrl, SortedSet<String> importedLogFileUrls) throws IOException {
        log.info("Downloading directory listings from {}.", (Object)baseUrl);
        ArrayList<String> directoryListings = new ArrayList<String>();
        directoryListings.add(baseUrl);
        TreeSet<String> newLogFileUrls = new TreeSet<String>();
        while (!directoryListings.isEmpty()) {
            String urlString = (String)directoryListings.remove(0);
            if (urlString.endsWith("/")) {
                directoryListings.addAll(Main.downloadDirectoryListing(urlString));
                continue;
            }
            if (!urlString.endsWith(".xz")) {
                log.debug("Skipping unrecognized URL {}.", (Object)urlString);
                continue;
            }
            if (importedLogFileUrls.contains(urlString)) continue;
            newLogFileUrls.add(urlString);
        }
        log.info("Found {} URLs of log files that have not yet been imported.", (Object)newLogFileUrls.size());
        return newLogFileUrls;
    }

    static List<String> downloadDirectoryListing(String urlString) throws IOException {
        log.debug("Downloading directory listing from {}.", (Object)urlString);
        ArrayList<String> urlStrings = new ArrayList<String>();
        try (BufferedReader br = new BufferedReader(new InputStreamReader(new URL(urlString).openStream()));){
            String line;
            while ((line = br.readLine()) != null) {
                Matcher matcher = URL_STRING_PATTERN.matcher(line);
                if (!matcher.matches() || matcher.group(1).startsWith("/")) continue;
                urlStrings.add(urlString + matcher.group(1));
            }
        }
        return urlStrings;
    }

    static void importLogFiles(Connection connection, SortedSet<String> newLogFileUrls) {
        log.info("Downloading, parsing, and importing {} log files.", (Object)newLogFileUrls.size());
        for (String urlString : newLogFileUrls) {
            try {
                Object[] metaData = Main.parseMetaData(urlString);
                if (metaData == null) continue;
                Map<String, Integer> parsedLogLines = Main.downloadAndParseLogFile(urlString);
                Main.importLogLines(connection, urlString, metaData, parsedLogLines);
            }
            catch (IOException | ParseException exc) {
                log.warn("Cannot download or parse log file with URL {}.  Retrying in the next run.", (Object)urlString, (Object)exc);
            }
            catch (SQLException exc) {
                log.warn("Cannot import log file with URL {} into the database.  Rolling back and retrying in the next run.", (Object)urlString, (Object)exc);
                try {
                    connection.rollback();
                }
                catch (SQLException sQLException) {}
            }
        }
    }

    private static Object[] parseMetaData(String urlString) throws ParseException {
        log.debug("Importing log file {}.", (Object)urlString);
        if (urlString.contains("-ssl-access.log-")) {
            log.debug("Skipping log file containing SSL requests with URL {}.", (Object)urlString);
            return null;
        }
        Matcher logFileUrlMatcher = LOG_FILE_URL_PATTERN.matcher(urlString);
        if (!logFileUrlMatcher.matches()) {
            log.debug("Skipping log file with unrecognized URL {}.", (Object)urlString);
            return null;
        }
        String server = logFileUrlMatcher.group(1);
        String site = logFileUrlMatcher.group(2);
        long logDateMillis = logDateFormat.parse(logFileUrlMatcher.group(3)).getTime();
        return new Object[]{server, site, logDateMillis};
    }

    static Map<String, Integer> downloadAndParseLogFile(String urlString) throws IOException {
        int skippedLines = 0;
        HashMap<String, Integer> parsedLogLines = new HashMap<String, Integer>();
        try (BufferedReader br = new BufferedReader(new InputStreamReader(new XZCompressorInputStream(new URL(urlString).openStream())));){
            String line;
            while ((line = br.readLine()) != null) {
                if (Main.parseLogLine(line, parsedLogLines)) continue;
                ++skippedLines;
            }
        }
        if (skippedLines > 0) {
            log.debug("Skipped {} lines while parsing log file {}.", (Object)skippedLines, (Object)urlString);
        }
        return parsedLogLines;
    }

    static boolean parseLogLine(String logLine, Map<String, Integer> parsedLogLines) {
        Matcher logLineMatcher = LOG_LINE_PATTERN.matcher(logLine);
        if (!logLineMatcher.matches()) {
            return false;
        }
        String method = logLineMatcher.group(1);
        String resource = logLineMatcher.group(2);
        int responseCode = Integer.parseInt(logLineMatcher.group(3));
        String combined = String.format("%s %s %d", method, resource, responseCode);
        if (!parsedLogLines.containsKey(combined)) {
            parsedLogLines.put(combined, 1);
        } else {
            parsedLogLines.put(combined, parsedLogLines.get(combined) + 1);
        }
        return true;
    }

    private static void importLogLines(Connection connection, String urlString, Object[] metaData, Map<String, Integer> parsedLogLines) throws SQLException {
        PreparedStatement psFiles = connection.prepareStatement("INSERT INTO files (url, server, site, log_date) VALUES (?, ?, ?, ?)", 1);
        PreparedStatement psResourcesSelect = connection.prepareStatement("SELECT resource_id FROM resources WHERE resource_string = ?");
        PreparedStatement psResourcesInsert = connection.prepareStatement("INSERT INTO resources (resource_string) VALUES (?)", 1);
        PreparedStatement psRequests = connection.prepareStatement("INSERT INTO requests (file_id, method, resource_id, response_code, count) VALUES (?, CAST(? AS method), ?, ?, ?)");
        String server = (String)metaData[0];
        String site = (String)metaData[1];
        long logDateMillis = (Long)metaData[2];
        int fileId = Main.insertFile(psFiles, urlString, server, site, logDateMillis);
        if (fileId < 0) {
            log.debug("Skipping previously imported log file {}.", (Object)urlString);
            return;
        }
        for (Map.Entry<String, Integer> requests : parsedLogLines.entrySet()) {
            String[] keyParts = requests.getKey().split(" ");
            String method = keyParts[0];
            String resource = keyParts[1];
            int responseCode = Integer.parseInt(keyParts[2]);
            int count = requests.getValue();
            int resourceId = Main.insertResource(psResourcesSelect, psResourcesInsert, resource);
            if (resourceId < 0) {
                log.error("Could not retrieve auto-generated key for new resources entry.");
                connection.rollback();
                return;
            }
            Main.insertRequest(psRequests, fileId, method, resourceId, responseCode, count);
        }
        connection.commit();
        log.debug("Finished importing log file with URL {} into database.", (Object)urlString);
    }

    private static int insertFile(PreparedStatement psFiles, String urlString, String server, String site, long logDateMillis) throws SQLException {
        int fileId = -1;
        psFiles.clearParameters();
        psFiles.setString(1, Main.truncateString(urlString, 2048));
        psFiles.setString(2, Main.truncateString(server, 32));
        psFiles.setString(3, Main.truncateString(site, 128));
        psFiles.setDate(4, new Date(logDateMillis));
        psFiles.execute();
        try (ResultSet rs = psFiles.getGeneratedKeys();){
            if (rs.next()) {
                fileId = rs.getInt(1);
            }
        }
        return fileId;
    }

    private static void insertRequest(PreparedStatement psRequests, int fileId, String method, int resourceId, int responseCode, int count) throws SQLException {
        psRequests.clearParameters();
        psRequests.setInt(1, fileId);
        psRequests.setString(2, method);
        psRequests.setInt(3, resourceId);
        psRequests.setInt(4, responseCode);
        psRequests.setInt(5, count);
        psRequests.execute();
    }

    private static int insertResource(PreparedStatement psResourcesSelect, PreparedStatement psResourcesInsert, String resource) throws SQLException {
        int resourceId = -1;
        String truncatedResource = Main.truncateString(resource, 2048);
        psResourcesSelect.clearParameters();
        psResourcesSelect.setString(1, truncatedResource);
        try (ResultSet rs = psResourcesSelect.executeQuery();){
            if (rs.next()) {
                resourceId = rs.getInt(1);
            }
        }
        if (resourceId < 0) {
            psResourcesInsert.clearParameters();
            psResourcesInsert.setString(1, truncatedResource);
            psResourcesInsert.execute();
            rs = psResourcesInsert.getGeneratedKeys();
            var6_6 = null;
            try {
                if (rs.next()) {
                    resourceId = rs.getInt(1);
                }
            }
            catch (Throwable throwable) {
                var6_6 = throwable;
                throw throwable;
            }
            finally {
                if (rs != null) {
                    if (var6_6 != null) {
                        try {
                            rs.close();
                        }
                        catch (Throwable throwable) {
                            var6_6.addSuppressed(throwable);
                        }
                    } else {
                        rs.close();
                    }
                }
            }
        }
        return resourceId;
    }

    private static String truncateString(String originalString, int truncateAfter) {
        if (originalString.length() > truncateAfter) {
            originalString = originalString.substring(0, truncateAfter);
        }
        return originalString;
    }

    static SortedSet<String> queryWebstats(Connection connection) throws SQLException {
        log.info("Querying statistics from database.");
        TreeSet<String> statistics = new TreeSet<String>();
        Statement st = connection.createStatement();
        String queryString = "SELECT log_date,request_type,platform,channel,locale,incremental,count FROM webstats";
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd", Locale.US);
        dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        Calendar calendar = Calendar.getInstance(TimeZone.getTimeZone("UTC"), Locale.US);
        try (ResultSet rs = st.executeQuery(queryString);){
            while (rs.next()) {
                statistics.add(String.format("%s,%s,%s,%s,%s,%s,%d", dateFormat.format(rs.getDate(LOG_DATE, calendar)), Main.emptyNull(rs.getString(REQUEST_TYPE)), Main.emptyNull(rs.getString(PLATFORM)), Main.emptyNull(rs.getString(CHANNEL)), Main.emptyNull(rs.getString(LOCALE)), Main.emptyNull(rs.getString(INCREMENTAL)), rs.getLong(COUNT)));
            }
        }
        return statistics;
    }

    private static String emptyNull(String text) {
        return null == text ? "" : text;
    }

    static void writeStatistics(Path webstatsPath, SortedSet<String> statistics) throws IOException {
        webstatsPath.toFile().getParentFile().mkdirs();
        ArrayList<String> lines = new ArrayList<String>();
        lines.add(ALL_COLUMNS);
        lines.addAll(statistics);
        log.info("Writing {} lines to {}.", (Object)lines.size(), (Object)webstatsPath.toFile().getAbsolutePath());
        Files.write(webstatsPath, lines, StandardCharsets.UTF_8, new OpenOption[0]);
    }

    private static void disconnectFromDatabase(Connection connection) throws SQLException {
        log.info("Disconnecting from database.");
        connection.close();
    }

    static {
        logDateFormat.setLenient(false);
        logDateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        LOG_LINE_PATTERN = Pattern.compile("^0.0.0.[01] - - \\[\\d{2}/\\w{3}/\\d{4}:00:00:00 \\+0000\\] \"(GET|HEAD) ([^ ]{1,2048}) HTTP[^ ]+\" (\\d+) (-|\\d+) \"-\" \"-\" -$");
    }
}

