/* Copyright 2015--2017 The Tor Project
 * See LICENSE for licensing information */

package org.torproject.descriptor.impl;

import static org.junit.Assert.assertEquals;

import org.torproject.descriptor.DescriptorParseException;
import org.torproject.descriptor.Microdescriptor;

import org.junit.Test;

public class MicrodescriptorImplTest {

  /* Helper class to build a microdescriptor based on default data and
   * modifications requested by test methods. */
  private static class DescriptorBuilder {

    private String onionKeyLines = "onion-key\n"
        + "-----BEGIN RSA PUBLIC KEY-----\n"
        + "MIGJAoGBALNZ4pNsHHkl7a+kFWbBmPHNAepjjvuhjTr1TaMB3UKuCRaXJmS2Qr"
        + "CW\nkTmINqdQUccwb3ghb7EBZfDtCUvjcwMSEsRRTVIZqVQsYj6m3n1CegOc4o"
        + "UutXaZ\nfkyty5XOgV4Qucx9wokzTMCHlO0V0x9y0FwFsK5Nb6ugqfQLLQ6XAg"
        + "MBAAE=\n"
        + "-----END RSA PUBLIC KEY-----";

    private static Microdescriptor createWithDefaultLines()
        throws DescriptorParseException {
      DescriptorBuilder db = new DescriptorBuilder();
      return db.buildDescriptor(true);
    }

    private String ntorOnionKeyLine =
        "ntor-onion-key PXLa7IGE+TzPDMsM5j9rFnDa37rd6kfZa5QuzqqJukw=";

    private String idLine = "id rsa1024 bvegfGxp8k7T9QFpjPTrPaJTa/8";

    private static Microdescriptor createWithIdLine(String line)
        throws DescriptorParseException {
      DescriptorBuilder db = new DescriptorBuilder();
      db.idLine = line;
      return db.buildDescriptor(true);
    }

    private byte[] buildDescriptorBytes() {
      StringBuilder sb = new StringBuilder();
      if (this.onionKeyLines != null) {
        sb.append(this.onionKeyLines).append("\n");
      }
      if (this.ntorOnionKeyLine != null) {
        sb.append(this.ntorOnionKeyLine).append("\n");
      }
      if (this.idLine != null) {
        sb.append(this.idLine).append("\n");
      }
      return sb.toString().getBytes();
    }

    private Microdescriptor buildDescriptor(
        boolean failUnrecognizedDescriptorLines)
        throws DescriptorParseException {
      byte[] descriptorBytes = this.buildDescriptorBytes();
      return new MicrodescriptorImpl(descriptorBytes,
          new int[] { 0, descriptorBytes.length },
          failUnrecognizedDescriptorLines);
    }
  }

  @Test()
  public void testDefaults() throws DescriptorParseException {
    Microdescriptor micro = DescriptorBuilder.createWithDefaultLines();
    assertEquals("ER1AC4KqT//o3pJDrqlmej5G2qW1EQYEr/IrMQHNc6I",
        micro.getDigestSha256Base64());
  }

  @Test(expected = DescriptorParseException.class)
  public void testIdRsa1024TooShort() throws DescriptorParseException {
    DescriptorBuilder.createWithIdLine("id rsa1024 AAAA");
  }

  @Test(expected = DescriptorParseException.class)
  public void testIdRsa1024TooLong() throws DescriptorParseException {
    DescriptorBuilder.createWithIdLine("id ed25519 AAAAAAAAAAAAAAAAAAAAAA"
        + "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA");
  }

  @Test(expected = DescriptorParseException.class)
  public void testIdRsa512() throws DescriptorParseException {
    DescriptorBuilder.createWithIdLine("id rsa512 "
        + "bvegfGxp8k7T9QFpjPTrPaJTa/8");
  }

  @Test(expected = DescriptorParseException.class)
  public void testIdEd25519Duplicate() throws DescriptorParseException {
    DescriptorBuilder.createWithIdLine(
        "id rsa1024 bvegfGxp8k7T9QFpjPTrPaJTa/8\n"
        + "id rsa1024 bvegfGxp8k7T9QFpjPTrPaJTa/8");
  }
}
