/*
 * Decompiled with CFR 0.152.
 */
package org.torproject.descriptor.impl;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.SortedMap;
import org.junit.Assert;
import org.junit.Test;
import org.torproject.descriptor.BandwidthHistory;
import org.torproject.descriptor.DescriptorParseException;
import org.torproject.descriptor.ServerDescriptor;
import org.torproject.descriptor.impl.RelayServerDescriptorImpl;

public class ServerDescriptorImplTest {
    private static final String IDENTITY_ED25519_LINES = "identity-ed25519\n-----BEGIN ED25519 CERT-----\nAQQABiX1AVGv5BuzJroQXbOh6vv1nbwc5rh2S13PyRFuLhTiifK4AQAgBACBCMwr\n4qgIlFDIzoC9ieJOtSkwrK+yXJPKlP8ojvgkx8cGKvhokOwA1eYDombzfwHcJ1EV\nbhEn/6g8i7wzO3LoqefIUrSAeEExOAOmm5mNmUIzL8EtnT6JHCr/sqUTUgA=\n-----END ED25519 CERT-----";
    private static final String MASTER_KEY_ED25519_LINE = "master-key-ed25519 gQjMK+KoCJRQyM6AvYniTrUpMKyvslyTypT/KI74JMc";
    private static final String ROUTER_SIG_ED25519_LINE = "router-sig-ed25519 y7WF9T2GFwkSDPZEhB55HgquIFOl5uXUFMYJPq3CXXUTKeJkSrtaZUB5s34fWdHQNtl84mH4dVaFMunHnwgYAw";
    private static final String ONION_KEY_CROSSCERT_LINES = "onion-key-crosscert\n-----BEGIN CROSSCERT-----\ngVWpiNgG2FekW1uonr4KKoqykjr4bqUBKGZfu6s9rvsV1TThnquZNP6ZhX2IPdQA\nlfKtzFggGu/4BiJ5oTSDj2sK2DMjY3rjrMQZ3I/wJ25yhc9gxjqYqUYO9MmJwALp\nfYkqp/t4WchJpyva/4hK8vITsI6eT2BfY/DWMy/suIE=\n-----END CROSSCERT-----";
    private static final String NTOR_ONION_KEY_CROSSCERT_LINES = "ntor-onion-key-crosscert 1\n-----BEGIN ED25519 CERT-----\nAQoABiUeAdauu1MxYGMmGLTCPaoes0RvW7udeLc1t8LZ4P3CDo5bAN4nrRfbCfOt\nz2Nwqn8tER1a+Ry6Vs+ilMZA55Rag4+f6Zdb1fmHWknCxbQlLHpqHACMtemPdaKa\nErPtMuiEqAc=\n-----END ED25519 CERT-----";

    @Test
    public void testSampleDescriptor() throws DescriptorParseException {
        DescriptorBuilder db = new DescriptorBuilder();
        RelayServerDescriptorImpl descriptor = new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        Assert.assertEquals((Object)"saberrider2008", (Object)descriptor.getNickname());
        Assert.assertEquals((Object)"94.134.192.243", (Object)descriptor.getAddress());
        Assert.assertEquals((long)9001L, (long)descriptor.getOrPort());
        Assert.assertEquals((long)0L, (long)descriptor.getSocksPort());
        Assert.assertEquals((long)0L, (long)descriptor.getDirPort());
        Assert.assertEquals((Object)"Tor 0.2.2.35 (git-b04388f9e7546a9f) on Linux i686", (Object)descriptor.getPlatform());
        Assert.assertEquals(Arrays.asList(1, 2), descriptor.getLinkProtocolVersions());
        Assert.assertEquals(Arrays.asList(1), descriptor.getCircuitProtocolVersions());
        Assert.assertEquals((long)1325390599000L, (long)descriptor.getPublishedMillis());
        Assert.assertEquals((Object)"D8733048FC8EC9102466AD8F3098622BF1BF71FD", (Object)descriptor.getFingerprint());
        Assert.assertEquals((long)48L, (long)descriptor.getUptime());
        Assert.assertEquals((long)51200L, (long)descriptor.getBandwidthRate());
        Assert.assertEquals((long)51200L, (long)descriptor.getBandwidthBurst());
        Assert.assertEquals((long)53470L, (long)descriptor.getBandwidthObserved());
        Assert.assertEquals((Object)"1469D1550738A25B1E7B47CDDBCD7B2899F51B74", (Object)descriptor.getExtraInfoDigest());
        Assert.assertEquals(Arrays.asList(2), descriptor.getHiddenServiceDirVersions());
        Assert.assertEquals((Object)"Random Person <nobody AT example dot com>", (Object)descriptor.getContact());
        Assert.assertEquals(Arrays.asList("reject *:*"), descriptor.getExitPolicyLines());
        Assert.assertFalse((boolean)descriptor.isHibernating());
        Assert.assertNull(descriptor.getFamilyEntries());
        Assert.assertNull((Object)descriptor.getReadHistory());
        Assert.assertNull((Object)descriptor.getWriteHistory());
        Assert.assertFalse((boolean)descriptor.getUsesEnhancedDnsLogic());
        Assert.assertFalse((boolean)descriptor.getCachesExtraInfo());
        Assert.assertFalse((boolean)descriptor.getAllowSingleHopExits());
        Assert.assertTrue((boolean)descriptor.getUnrecognizedLines().isEmpty());
    }

    @Test(expected=DescriptorParseException.class)
    public void testRouterLineMissing() throws DescriptorParseException {
        DescriptorBuilder.createWithRouterLine(null);
    }

    @Test
    public void testRouterOpt() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithRouterLine("opt router saberrider2008 94.134.192.243 9001 0 0");
        Assert.assertEquals((Object)"saberrider2008", (Object)descriptor.getNickname());
        Assert.assertEquals((Object)"94.134.192.243", (Object)descriptor.getAddress());
        Assert.assertEquals((long)9001L, (long)descriptor.getOrPort());
        Assert.assertEquals((long)0L, (long)descriptor.getSocksPort());
        Assert.assertEquals((long)0L, (long)descriptor.getDirPort());
    }

    @Test(expected=DescriptorParseException.class)
    public void testRouterLinePrecedingHibernatingLine() throws DescriptorParseException {
        DescriptorBuilder.createWithRouterLine("hibernating 1\nrouter saberrider2008 94.134.192.243 9001 0 0");
    }

    @Test(expected=DescriptorParseException.class)
    public void testNicknameMissing() throws DescriptorParseException {
        DescriptorBuilder.createWithRouterLine("router  94.134.192.243 9001 0 0");
    }

    @Test(expected=DescriptorParseException.class)
    public void testNicknameInvalidChar() throws DescriptorParseException {
        DescriptorBuilder.createWithRouterLine("router $aberrider2008 94.134.192.243 9001 0 0");
    }

    @Test(expected=DescriptorParseException.class)
    public void testNicknameTooLong() throws DescriptorParseException {
        DescriptorBuilder.createWithRouterLine("router saberrider2008ReallyLongNickname 94.134.192.243 9001 0 0");
    }

    @Test
    public void testNicknameTwoSpaces() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithRouterLine("router saberrider2008  94.134.192.243 9001 0 0");
        Assert.assertEquals((Object)"saberrider2008", (Object)descriptor.getNickname());
        Assert.assertEquals((Object)"94.134.192.243", (Object)descriptor.getAddress());
    }

    @Test(expected=DescriptorParseException.class)
    public void testAddress24() throws DescriptorParseException {
        DescriptorBuilder.createWithRouterLine("router saberrider2008 94.134.192/24 9001 0 0");
    }

    @Test(expected=DescriptorParseException.class)
    public void testAddress294() throws DescriptorParseException {
        DescriptorBuilder.createWithRouterLine("router saberrider2008 294.134.192.243 9001 0 0");
    }

    @Test(expected=DescriptorParseException.class)
    public void testAddressMissing() throws DescriptorParseException {
        DescriptorBuilder.createWithRouterLine("router saberrider2008  9001 0 0");
    }

    @Test(expected=DescriptorParseException.class)
    public void testOrPort99001() throws DescriptorParseException {
        DescriptorBuilder.createWithRouterLine("router saberrider2008 94.134.192.243 99001 0 0");
    }

    @Test(expected=DescriptorParseException.class)
    public void testOrPortMissing() throws DescriptorParseException {
        DescriptorBuilder.createWithRouterLine("router saberrider2008 94.134.192.243  0 0");
    }

    @Test(expected=DescriptorParseException.class)
    public void testOrPortOne() throws DescriptorParseException {
        DescriptorBuilder.createWithRouterLine("router saberrider2008 94.134.192.243 one 0 0");
    }

    @Test(expected=DescriptorParseException.class)
    public void testOrPortNewline() throws DescriptorParseException {
        DescriptorBuilder.createWithRouterLine("router saberrider2008 94.134.192.243 0\n 0 0");
    }

    @Test(expected=DescriptorParseException.class)
    public void testDirPortMissing() throws DescriptorParseException {
        DescriptorBuilder.createWithRouterLine("router saberrider2008 94.134.192.243 9001 0 ");
    }

    @Test
    public void testPlatformMissing() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithPlatformLine(null);
        Assert.assertNull((Object)descriptor.getPlatform());
    }

    @Test
    public void testPlatformOpt() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithPlatformLine("opt platform Tor 0.2.2.35 (git-b04388f9e7546a9f) on Linux i686");
        Assert.assertEquals((Object)"Tor 0.2.2.35 (git-b04388f9e7546a9f) on Linux i686", (Object)descriptor.getPlatform());
    }

    @Test
    public void testPlatformNoSpace() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithPlatformLine("platform");
        Assert.assertEquals((Object)"", (Object)descriptor.getPlatform());
    }

    @Test
    public void testPlatformSpace() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithPlatformLine("platform ");
        Assert.assertEquals((Object)"", (Object)descriptor.getPlatform());
    }

    @Test
    public void testProtocolsNoOpt() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithProtocolsLine("protocols Link 1 2 Circuit 1");
        Assert.assertEquals(Arrays.asList(1, 2), descriptor.getLinkProtocolVersions());
        Assert.assertEquals(Arrays.asList(1), descriptor.getCircuitProtocolVersions());
    }

    @Test(expected=DescriptorParseException.class)
    public void testProtocolsAB() throws DescriptorParseException {
        DescriptorBuilder.createWithProtocolsLine("opt protocols Link A B Circuit 1");
    }

    @Test(expected=DescriptorParseException.class)
    public void testProtocolsNoCircuitVersions() throws DescriptorParseException {
        DescriptorBuilder.createWithProtocolsLine("opt protocols Link 1 2");
    }

    @Test(expected=DescriptorParseException.class)
    public void testPublishedMissing() throws DescriptorParseException {
        DescriptorBuilder.createWithPublishedLine(null);
    }

    @Test
    public void testPublishedOpt() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithPublishedLine("opt published 2012-01-01 04:03:19");
        Assert.assertEquals((long)1325390599000L, (long)descriptor.getPublishedMillis());
    }

    @Test(expected=DescriptorParseException.class)
    public void testPublished2039() throws DescriptorParseException {
        DescriptorBuilder.createWithPublishedLine("published 2039-01-01 04:03:19");
    }

    @Test(expected=DescriptorParseException.class)
    public void testPublished1912() throws DescriptorParseException {
        DescriptorBuilder.createWithPublishedLine("published 1912-01-01 04:03:19");
    }

    @Test(expected=DescriptorParseException.class)
    public void testPublishedFeb31() throws DescriptorParseException {
        DescriptorBuilder.createWithPublishedLine("published 2012-02-31 04:03:19");
    }

    @Test(expected=DescriptorParseException.class)
    public void testPublishedNoTime() throws DescriptorParseException {
        DescriptorBuilder.createWithPublishedLine("published 2012-01-01");
    }

    @Test
    public void testPublishedMillis() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithPublishedLine("opt published 2012-01-01 04:03:19.123");
        Assert.assertEquals((long)1325390599000L, (long)descriptor.getPublishedMillis());
    }

    @Test
    public void testFingerprintNoOpt() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithFingerprintLine("fingerprint D873 3048 FC8E C910 2466 AD8F 3098 622B F1BF 71FD");
        Assert.assertEquals((Object)"D8733048FC8EC9102466AD8F3098622BF1BF71FD", (Object)descriptor.getFingerprint());
    }

    @Test(expected=DescriptorParseException.class)
    public void testFingerprintG() throws DescriptorParseException {
        DescriptorBuilder.createWithFingerprintLine("opt fingerprint G873 3048 FC8E C910 2466 AD8F 3098 622B F1BF 71FD");
    }

    @Test(expected=DescriptorParseException.class)
    public void testFingerprintTooShort() throws DescriptorParseException {
        DescriptorBuilder.createWithFingerprintLine("opt fingerprint D873 3048 FC8E C910 2466 AD8F 3098 622B F1BF");
    }

    @Test(expected=DescriptorParseException.class)
    public void testFingerprintTooLong() throws DescriptorParseException {
        DescriptorBuilder.createWithFingerprintLine("opt fingerprint D873 3048 FC8E C910 2466 AD8F 3098 622B F1BF 71FD D873");
    }

    @Test(expected=DescriptorParseException.class)
    public void testFingerprintNoSpaces() throws DescriptorParseException {
        DescriptorBuilder.createWithFingerprintLine("opt fingerprint D8733048FC8EC9102466AD8F3098622BF1BF71FD");
    }

    @Test
    public void testUptimeMissing() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithUptimeLine(null);
        Assert.assertNull((Object)descriptor.getUptime());
    }

    @Test
    public void testUptimeOpt() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithUptimeLine("opt uptime 48");
        Assert.assertEquals((long)48L, (long)descriptor.getUptime());
    }

    @Test(expected=DescriptorParseException.class)
    public void testUptimeFourtyEight() throws DescriptorParseException {
        DescriptorBuilder.createWithUptimeLine("uptime fourty-eight");
    }

    @Test
    public void testUptimeMinusOne() throws DescriptorParseException {
        DescriptorBuilder.createWithUptimeLine("uptime -1");
    }

    @Test(expected=DescriptorParseException.class)
    public void testUptimeSpace() throws DescriptorParseException {
        DescriptorBuilder.createWithUptimeLine("uptime ");
    }

    @Test(expected=DescriptorParseException.class)
    public void testUptimeNoSpace() throws DescriptorParseException {
        DescriptorBuilder.createWithUptimeLine("uptime");
    }

    @Test(expected=DescriptorParseException.class)
    public void testUptimeFourEight() throws DescriptorParseException {
        DescriptorBuilder.createWithUptimeLine("uptime 4 8");
    }

    @Test
    public void testBandwidthOpt() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithBandwidthLine("opt bandwidth 51200 51200 53470");
        Assert.assertEquals((long)51200L, (long)descriptor.getBandwidthRate());
        Assert.assertEquals((long)51200L, (long)descriptor.getBandwidthBurst());
        Assert.assertEquals((long)53470L, (long)descriptor.getBandwidthObserved());
    }

    @Test(expected=DescriptorParseException.class)
    public void testBandwidthMissing() throws DescriptorParseException {
        DescriptorBuilder.createWithBandwidthLine(null);
    }

    @Test(expected=DescriptorParseException.class)
    public void testBandwidthOneValue() throws DescriptorParseException {
        DescriptorBuilder.createWithBandwidthLine("bandwidth 51200");
    }

    @Test
    public void testBandwidthTwoValues() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithBandwidthLine("bandwidth 51200 51200");
        Assert.assertEquals((long)51200L, (long)descriptor.getBandwidthRate());
        Assert.assertEquals((long)51200L, (long)descriptor.getBandwidthBurst());
        Assert.assertEquals((long)-1L, (long)descriptor.getBandwidthObserved());
    }

    @Test(expected=DescriptorParseException.class)
    public void testBandwidthFourValues() throws DescriptorParseException {
        DescriptorBuilder.createWithBandwidthLine("bandwidth 51200 51200 53470 53470");
    }

    @Test(expected=DescriptorParseException.class)
    public void testBandwidthMinusOneTwoThree() throws DescriptorParseException {
        DescriptorBuilder.createWithBandwidthLine("bandwidth -1 -2 -3");
    }

    @Test
    public void testExtraInfoDigestNoOpt() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithExtraInfoDigestLine("extra-info-digest 1469D1550738A25B1E7B47CDDBCD7B2899F51B74");
        Assert.assertEquals((Object)"1469D1550738A25B1E7B47CDDBCD7B2899F51B74", (Object)descriptor.getExtraInfoDigest());
    }

    @Test(expected=DescriptorParseException.class)
    public void testExtraInfoDigestNoSpace() throws DescriptorParseException {
        DescriptorBuilder.createWithExtraInfoDigestLine("opt extra-info-digest");
    }

    @Test(expected=DescriptorParseException.class)
    public void testExtraInfoDigestTooShort() throws DescriptorParseException {
        DescriptorBuilder.createWithExtraInfoDigestLine("opt extra-info-digest 1469D1550738A25B1E7B47CDDBCD7B2899F5");
    }

    @Test(expected=DescriptorParseException.class)
    public void testExtraInfoDigestTooLong() throws DescriptorParseException {
        DescriptorBuilder.createWithExtraInfoDigestLine("opt extra-info-digest 1469D1550738A25B1E7B47CDDBCD7B2899F51B741469");
    }

    @Test
    public void testExtraInfoDigestMissing() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithExtraInfoDigestLine(null);
        Assert.assertNull((Object)descriptor.getExtraInfoDigest());
    }

    @Test
    public void testExtraInfoDigestAdditionalDigest() throws DescriptorParseException {
        String extraInfoDigest = "0879DB7B765218D7B3AE7557669D20307BB21CAA";
        String additionalExtraInfoDigest = "V609l+N6ActBveebfNbH5lQ6wHDNstDkFgyqEhBHwtA";
        String extraInfoDigestLine = String.format("extra-info-digest %s %s", extraInfoDigest, additionalExtraInfoDigest);
        ServerDescriptor descriptor = DescriptorBuilder.createWithExtraInfoDigestLine(extraInfoDigestLine);
        Assert.assertEquals((Object)extraInfoDigest, (Object)descriptor.getExtraInfoDigest());
    }

    @Test
    public void testOnionKeyOpt() throws DescriptorParseException {
        DescriptorBuilder.createWithOnionKeyLines("opt onion-key\n-----BEGIN RSA PUBLIC KEY-----\nMIGJAoGBAKM+iiHhO6eHsvd6Xjws9z9EQB1V/Bpuy5ciGJ1U4V9SeiKooSo5BpPL\no3XT+6PIgzl3R6uycjS3Ejk47vLEJdcVTm/VG6E0ppu3olIynCI4QryfCEuC3cTF\n9wE4WXY4nX7w0RTN18UVLxrt1A9PP0cobFNiPs9rzJCbKFfacOkpAgMBAAE=\n-----END RSA PUBLIC KEY-----");
    }

    @Test
    public void testSigningKeyOpt() throws DescriptorParseException {
        DescriptorBuilder.createWithSigningKeyLines("opt signing-key\n-----BEGIN RSA PUBLIC KEY-----\nMIGJAoGBALMm3r3QDh482Ewe6Ub9wvRIfmEkoNX6q5cEAtQRNHSDcNx41gjELbcl\nEniVMParBYACKfOxkS+mTTnIRDKVNEJTsDOwryNrc4X9JnPc/nn6ymYPiNDhUROG\n8URDIhQoixcUeyyrVB8sxliSstKimulGnB7xpjYOlO8JKaHLNL4TAgMBAAE=\n-----END RSA PUBLIC KEY-----");
    }

    @Test
    public void testHiddenServiceDirMissing() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithHiddenServiceDirLine(null);
        Assert.assertNull(descriptor.getHiddenServiceDirVersions());
    }

    @Test
    public void testHiddenServiceDirNoOpt() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithHiddenServiceDirLine("hidden-service-dir");
        Assert.assertEquals(Arrays.asList(2), descriptor.getHiddenServiceDirVersions());
    }

    @Test
    public void testHiddenServiceDirVersions2And3() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithHiddenServiceDirLine("hidden-service-dir 2 3");
        Assert.assertEquals(Arrays.asList(2, 3), descriptor.getHiddenServiceDirVersions());
    }

    @Test
    public void testContactMissing() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithContactLine(null);
        Assert.assertNull((Object)descriptor.getContact());
    }

    @Test
    public void testContactOpt() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithContactLine("opt contact Random Person");
        Assert.assertEquals((Object)"Random Person", (Object)descriptor.getContact());
    }

    @Test(expected=DescriptorParseException.class)
    public void testContactDuplicate() throws DescriptorParseException {
        DescriptorBuilder.createWithContactLine("contact Random Person\ncontact Random Person");
    }

    @Test
    public void testContactNoSpace() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithContactLine("contact");
        Assert.assertEquals((Object)"", (Object)descriptor.getContact());
    }

    @Test
    public void testContactCarriageReturn() throws DescriptorParseException {
        String contactString = "Random Person -----BEGIN PGP PUBLIC KEY BLOCK-----\rVersion: GnuPG v1 dot 4 dot 7 (Darwin)\r\rmQGiBEbb0rcRBADqBiUXsmtpJifh74irNnkHbhKMj8O4TqenaZYhdjLWouZsZd07\rmTQoP40G4zqOrVEOOcXpdSiRnHWJYfgTnkibNZrOZEZLn3H1ywpovEgESmoGEdAX\roid3XuIYRpRnqoafbFg9sg+OofX/mGrO+5ACfagQ9rlfx2oxCWijYwpYFRk3NhCY=\r=Xaw3\r-----END PGP PUBLIC KEY BLOCK-----";
        ServerDescriptor descriptor = DescriptorBuilder.createWithContactLine("contact " + contactString);
        Assert.assertEquals((Object)contactString, (Object)descriptor.getContact());
    }

    @Test
    public void testExitPolicyRejectAllAcceptAll() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithExitPolicyLines("reject *:*\naccept *:*");
        Assert.assertEquals(Arrays.asList("reject *:*", "accept *:*"), descriptor.getExitPolicyLines());
    }

    @Test
    public void testExitPolicyOpt() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithExitPolicyLines("opt reject *:*");
        Assert.assertEquals(Arrays.asList("reject *:*"), descriptor.getExitPolicyLines());
    }

    @Test(expected=DescriptorParseException.class)
    public void testExitPolicyNoPort() throws DescriptorParseException {
        DescriptorBuilder.createWithExitPolicyLines("reject *");
    }

    @Test
    public void testExitPolicyAccept80RejectAll() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithExitPolicyLines("accept *:80\nreject *:*");
        Assert.assertEquals(Arrays.asList("accept *:80", "reject *:*"), descriptor.getExitPolicyLines());
    }

    @Test(expected=DescriptorParseException.class)
    public void testExitPolicyReject321() throws DescriptorParseException {
        DescriptorBuilder.createWithExitPolicyLines("reject 123.123.123.321:80");
    }

    @Test(expected=DescriptorParseException.class)
    public void testExitPolicyRejectPort66666() throws DescriptorParseException {
        DescriptorBuilder.createWithExitPolicyLines("reject *:66666");
    }

    @Test(expected=DescriptorParseException.class)
    public void testExitPolicyProjectAll() throws DescriptorParseException {
        DescriptorBuilder.createWithExitPolicyLines("project *:*");
    }

    @Test(expected=DescriptorParseException.class)
    public void testExitPolicyMissing() throws DescriptorParseException {
        DescriptorBuilder.createWithExitPolicyLines(null);
    }

    @Test
    public void testExitPolicyMaskTypes() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithExitPolicyLines("reject 192.168.0.0/16:*\nreject 94.134.192.243/255.255.255.0:*");
        Assert.assertEquals(Arrays.asList("reject 192.168.0.0/16:*", "reject 94.134.192.243/255.255.255.0:*"), descriptor.getExitPolicyLines());
    }

    @Test
    public void testRouterSignatureOpt() throws DescriptorParseException {
        DescriptorBuilder.createWithRouterSignatureLines("opt router-signature\n-----BEGIN SIGNATURE-----\ncrypto lines are ignored anyway\n-----END SIGNATURE-----");
    }

    @Test(expected=DescriptorParseException.class)
    public void testRouterSignatureNotLastLine() throws DescriptorParseException {
        DescriptorBuilder.createWithRouterSignatureLines("router-signature\n-----BEGIN SIGNATURE-----\no4j+kH8UQfjBwepUnr99v0ebN8RpzHJ/lqYsTojXHy9kMr1RNI9IDeSzA7PSqTuV\n4PL8QsGtlfwthtIoZpB2srZeyN/mcpA9fa1JXUrt/UN9K/+32Cyaad7h0nHE6Xfb\njqpXDpnBpvk4zjmzjjKYnIsUWTnADmu0fo3xTRqXi7g=\n-----END SIGNATURE-----\ncontact me");
    }

    @Test
    public void testHibernatingOpt() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithHibernatingLine("opt hibernating 1");
        Assert.assertTrue((boolean)descriptor.isHibernating());
    }

    @Test
    public void testHibernatingFalse() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithHibernatingLine("hibernating 0");
        Assert.assertFalse((boolean)descriptor.isHibernating());
    }

    @Test
    public void testHibernatingTrue() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithHibernatingLine("hibernating 1");
        Assert.assertTrue((boolean)descriptor.isHibernating());
    }

    @Test(expected=DescriptorParseException.class)
    public void testHibernatingYep() throws DescriptorParseException {
        DescriptorBuilder.createWithHibernatingLine("hibernating yep");
    }

    @Test(expected=DescriptorParseException.class)
    public void testHibernatingNoSpace() throws DescriptorParseException {
        DescriptorBuilder.createWithHibernatingLine("hibernating");
    }

    @Test
    public void testFamilyOpt() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithFamilyLine("opt family saberrider2008");
        Assert.assertEquals(Arrays.asList("saberrider2008"), descriptor.getFamilyEntries());
    }

    @Test
    public void testFamilyFingerprint() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithFamilyLine("family $D8733048FC8EC9102466AD8F3098622BF1BF71FD");
        Assert.assertEquals(Arrays.asList("$D8733048FC8EC9102466AD8F3098622BF1BF71FD"), descriptor.getFamilyEntries());
    }

    @Test
    public void testFamilyNickname() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithFamilyLine("family saberrider2008");
        Assert.assertEquals(Arrays.asList("saberrider2008"), descriptor.getFamilyEntries());
    }

    @Test(expected=DescriptorParseException.class)
    public void testFamilyDuplicate() throws DescriptorParseException {
        DescriptorBuilder.createWithFamilyLine("family saberrider2008\nfamily saberrider2008");
    }

    @Test(expected=DescriptorParseException.class)
    public void testFamilyNicknamePrefix() throws DescriptorParseException {
        DescriptorBuilder.createWithFamilyLine("family $saberrider2008");
    }

    @Test(expected=DescriptorParseException.class)
    public void testFamilyFingerprintNoPrefix() throws DescriptorParseException {
        DescriptorBuilder.createWithFamilyLine("family D8733048FC8EC9102466AD8F3098622BF1BF71FD");
    }

    @Test
    public void testFamilyFingerprintNicknameNamed() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithFamilyLine("family $D8733048FC8EC9102466AD8F3098622BF1BF71FD=saberrider2008");
        Assert.assertEquals(Arrays.asList("$D8733048FC8EC9102466AD8F3098622BF1BF71FD=saberrider2008"), descriptor.getFamilyEntries());
    }

    @Test
    public void testFamilyFingerprintNicknameUnnamed() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithFamilyLine("family $D8733048FC8EC9102466AD8F3098622BF1BF71FD~saberrider2008");
        Assert.assertEquals(Arrays.asList("$D8733048FC8EC9102466AD8F3098622BF1BF71FD~saberrider2008"), descriptor.getFamilyEntries());
    }

    @Test
    public void testWriteHistory() throws DescriptorParseException {
        String writeHistoryLine = "write-history 2012-01-01 03:51:44 (900 s) 4345856,261120,7591936,1748992";
        ServerDescriptor descriptor = DescriptorBuilder.createWithWriteHistoryLine(writeHistoryLine);
        Assert.assertNotNull((Object)descriptor.getWriteHistory());
        BandwidthHistory parsedWriteHistory = descriptor.getWriteHistory();
        Assert.assertEquals((Object)writeHistoryLine, (Object)parsedWriteHistory.getLine());
        Assert.assertEquals((long)1325389904000L, (long)parsedWriteHistory.getHistoryEndMillis());
        Assert.assertEquals((long)900L, (long)parsedWriteHistory.getIntervalLength());
        SortedMap<Long, Long> bandwidthValues = parsedWriteHistory.getBandwidthValues();
        Assert.assertEquals((long)4345856L, (long)((Long)bandwidthValues.remove(1325387204000L)));
        Assert.assertEquals((long)261120L, (long)((Long)bandwidthValues.remove(1325388104000L)));
        Assert.assertEquals((long)7591936L, (long)((Long)bandwidthValues.remove(1325389004000L)));
        Assert.assertEquals((long)1748992L, (long)((Long)bandwidthValues.remove(1325389904000L)));
        Assert.assertTrue((boolean)bandwidthValues.isEmpty());
    }

    @Test
    public void testWriteHistoryOpt() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithWriteHistoryLine("opt write-history 2012-01-01 03:51:44 (900 s) 4345856,261120,7591936,1748992");
        Assert.assertNotNull((Object)descriptor.getWriteHistory());
    }

    @Test(expected=DescriptorParseException.class)
    public void testWriteHistory3012() throws DescriptorParseException {
        DescriptorBuilder.createWithWriteHistoryLine("write-history 3012-01-01 03:51:44 (900 s) 4345856,261120,7591936,1748992");
    }

    @Test(expected=DescriptorParseException.class)
    public void testWriteHistoryNoSeconds() throws DescriptorParseException {
        DescriptorBuilder.createWithWriteHistoryLine("write-history 2012-01-01 03:51 (900 s) 4345856,261120,7591936,1748992");
    }

    @Test(expected=DescriptorParseException.class)
    public void testWriteHistoryNoParathenses() throws DescriptorParseException {
        DescriptorBuilder.createWithWriteHistoryLine("write-history 2012-01-01 03:51:44 900 s 4345856,261120,7591936,1748992");
    }

    @Test(expected=DescriptorParseException.class)
    public void testWriteHistoryNoSpaceSeconds() throws DescriptorParseException {
        DescriptorBuilder.createWithWriteHistoryLine("write-history 2012-01-01 03:51:44 (900s) 4345856,261120,7591936,1748992");
    }

    @Test(expected=DescriptorParseException.class)
    public void testWriteHistoryTrailingComma() throws DescriptorParseException {
        DescriptorBuilder.createWithWriteHistoryLine("write-history 2012-01-01 03:51:44 (900 s) 4345856,261120,7591936,");
    }

    @Test(expected=DescriptorParseException.class)
    public void testWriteHistoryOneTwoThree() throws DescriptorParseException {
        DescriptorBuilder.createWithWriteHistoryLine("write-history 2012-01-01 03:51:44 (900 s) one,two,three");
    }

    @Test
    public void testWriteHistoryNoValuesSpace() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithWriteHistoryLine("write-history 2012-01-01 03:51:44 (900 s) ");
        Assert.assertEquals((long)900L, (long)descriptor.getWriteHistory().getIntervalLength());
        Assert.assertTrue((boolean)descriptor.getWriteHistory().getBandwidthValues().isEmpty());
    }

    @Test
    public void testWriteHistoryNoValuesNoSpace() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithWriteHistoryLine("write-history 2012-01-01 03:51:44 (900 s)");
        Assert.assertEquals((long)900L, (long)descriptor.getWriteHistory().getIntervalLength());
        Assert.assertTrue((boolean)descriptor.getWriteHistory().getBandwidthValues().isEmpty());
    }

    @Test(expected=DescriptorParseException.class)
    public void testWriteHistoryNoS() throws DescriptorParseException {
        DescriptorBuilder.createWithWriteHistoryLine("write-history 2012-01-01 03:51:44 (900 ");
    }

    @Test(expected=DescriptorParseException.class)
    public void testWriteHistoryTrailingNumber() throws DescriptorParseException {
        DescriptorBuilder.createWithWriteHistoryLine("write-history 2012-01-01 03:51:44 (900 s) 4345856 1");
    }

    @Test
    public void testWriteHistory1800Seconds() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithWriteHistoryLine("write-history 2012-01-01 03:51:44 (1800 s) 4345856");
        Assert.assertEquals((long)1800L, (long)descriptor.getWriteHistory().getIntervalLength());
    }

    @Test
    public void testReadHistory() throws DescriptorParseException {
        String readHistoryLine = "read-history 2012-01-01 03:51:44 (900 s) 4268032,139264,7797760,1415168";
        ServerDescriptor descriptor = DescriptorBuilder.createWithReadHistoryLine(readHistoryLine);
        Assert.assertNotNull((Object)descriptor.getReadHistory());
        BandwidthHistory parsedReadHistory = descriptor.getReadHistory();
        Assert.assertEquals((Object)readHistoryLine, (Object)parsedReadHistory.getLine());
        Assert.assertEquals((long)1325389904000L, (long)parsedReadHistory.getHistoryEndMillis());
        Assert.assertEquals((long)900L, (long)parsedReadHistory.getIntervalLength());
        SortedMap<Long, Long> bandwidthValues = parsedReadHistory.getBandwidthValues();
        Assert.assertEquals((long)4268032L, (long)((Long)bandwidthValues.remove(1325387204000L)));
        Assert.assertEquals((long)139264L, (long)((Long)bandwidthValues.remove(1325388104000L)));
        Assert.assertEquals((long)7797760L, (long)((Long)bandwidthValues.remove(1325389004000L)));
        Assert.assertEquals((long)1415168L, (long)((Long)bandwidthValues.remove(1325389904000L)));
        Assert.assertTrue((boolean)bandwidthValues.isEmpty());
    }

    @Test
    public void testReadHistoryTwoSpaces() throws DescriptorParseException {
        String readHistoryLine = "opt read-history  2012-01-01 03:51:44 (900 s) 4268032,139264,7797760,1415168";
        DescriptorBuilder.createWithReadHistoryLine(readHistoryLine);
    }

    @Test
    public void testEventdnsOpt() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithEventdnsLine("opt eventdns 1");
        Assert.assertTrue((boolean)descriptor.getUsesEnhancedDnsLogic());
    }

    @Test
    public void testEventdns1() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithEventdnsLine("eventdns 1");
        Assert.assertTrue((boolean)descriptor.getUsesEnhancedDnsLogic());
    }

    @Test
    public void testEventdns0() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithEventdnsLine("eventdns 0");
        Assert.assertFalse((boolean)descriptor.getUsesEnhancedDnsLogic());
    }

    @Test(expected=DescriptorParseException.class)
    public void testEventdnsTrue() throws DescriptorParseException {
        DescriptorBuilder.createWithEventdnsLine("eventdns true");
    }

    @Test(expected=DescriptorParseException.class)
    public void testEventdnsNo() throws DescriptorParseException {
        DescriptorBuilder.createWithEventdnsLine("eventdns no");
    }

    @Test
    public void testCachesExtraInfoOpt() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithCachesExtraInfoLine("opt caches-extra-info");
        Assert.assertTrue((boolean)descriptor.getCachesExtraInfo());
    }

    @Test
    public void testCachesExtraInfoNoSpace() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithCachesExtraInfoLine("caches-extra-info");
        Assert.assertTrue((boolean)descriptor.getCachesExtraInfo());
    }

    @Test(expected=DescriptorParseException.class)
    public void testCachesExtraInfoTrue() throws DescriptorParseException {
        DescriptorBuilder.createWithCachesExtraInfoLine("caches-extra-info true");
    }

    @Test
    public void testAllowSingleHopExitsOpt() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithAllowSingleHopExitsLine("opt allow-single-hop-exits");
        Assert.assertTrue((boolean)descriptor.getAllowSingleHopExits());
    }

    @Test
    public void testAllowSingleHopExitsNoSpace() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithAllowSingleHopExitsLine("allow-single-hop-exits");
        Assert.assertTrue((boolean)descriptor.getAllowSingleHopExits());
    }

    @Test(expected=DescriptorParseException.class)
    public void testAllowSingleHopExitsTrue() throws DescriptorParseException {
        DescriptorBuilder.createWithAllowSingleHopExitsLine("allow-single-hop-exits true");
    }

    @Test(expected=DescriptorParseException.class)
    public void testAllowSingleHopExitsNonAsciiKeyword() throws DescriptorParseException {
        DescriptorBuilder.createWithNonAsciiLineBytes(new byte[]{20, -2, 24, 97, 108, 108, 111, 119, 45, 115, 105, 110, 103, 108, 101, 45, 104, 111, 112, 45, 101, 120, 105, 116, 115}, false);
    }

    @Test
    public void testIpv6PolicyLine() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithIpv6PolicyLine("ipv6-policy accept 80,1194,1220,1293");
        Assert.assertEquals((Object)"accept", (Object)descriptor.getIpv6DefaultPolicy());
        Assert.assertEquals((Object)"80,1194,1220,1293", (Object)descriptor.getIpv6PortList());
    }

    @Test(expected=DescriptorParseException.class)
    public void testIpv6PolicyLineNoPolicy() throws DescriptorParseException {
        DescriptorBuilder.createWithIpv6PolicyLine("ipv6-policy 80");
    }

    @Test(expected=DescriptorParseException.class)
    public void testIpv6PolicyLineNoPorts() throws DescriptorParseException {
        DescriptorBuilder.createWithIpv6PolicyLine("ipv6-policy accept");
    }

    @Test(expected=DescriptorParseException.class)
    public void testIpv6PolicyLineNoPolicyNoPorts() throws DescriptorParseException {
        DescriptorBuilder.createWithIpv6PolicyLine("ipv6-policy ");
    }

    @Test(expected=DescriptorParseException.class)
    public void testIpv6PolicyLineProject() throws DescriptorParseException {
        DescriptorBuilder.createWithIpv6PolicyLine("ipv6-policy project 80");
    }

    @Test(expected=DescriptorParseException.class)
    public void testTwoIpv6PolicyLines() throws DescriptorParseException {
        DescriptorBuilder.createWithIpv6PolicyLine("ipv6-policy accept 80,1194,1220,1293\nipv6-policy accept 80,1194,1220,1293");
    }

    @Test
    public void testNtorOnionKeyLine() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithNtorOnionKeyLine("ntor-onion-key Y/XgaHcPIJVa4D55kir9QLH8rEYAaLXuv3c3sm8jYhY=");
        Assert.assertEquals((Object)"Y/XgaHcPIJVa4D55kir9QLH8rEYAaLXuv3c3sm8jYhY", (Object)descriptor.getNtorOnionKey());
    }

    @Test
    public void testNtorOnionKeyLineNoPadding() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithNtorOnionKeyLine("ntor-onion-key Y/XgaHcPIJVa4D55kir9QLH8rEYAaLXuv3c3sm8jYhY");
        Assert.assertEquals((Object)"Y/XgaHcPIJVa4D55kir9QLH8rEYAaLXuv3c3sm8jYhY", (Object)descriptor.getNtorOnionKey());
    }

    @Test(expected=DescriptorParseException.class)
    public void testNtorOnionKeyLineNoKey() throws DescriptorParseException {
        DescriptorBuilder.createWithNtorOnionKeyLine("ntor-onion-key ");
    }

    @Test(expected=DescriptorParseException.class)
    public void testNtorOnionKeyLineTwoKeys() throws DescriptorParseException {
        DescriptorBuilder.createWithNtorOnionKeyLine("ntor-onion-key Y/XgaHcPIJVa4D55kir9QLH8rEYAaLXuv3c3sm8jYhY Y/XgaHcPIJVa4D55kir9QLH8rEYAaLXuv3c3sm8jYhY");
    }

    @Test(expected=DescriptorParseException.class)
    public void testTwoNtorOnionKeyLines() throws DescriptorParseException {
        DescriptorBuilder.createWithNtorOnionKeyLine("ntor-onion-key Y/XgaHcPIJVa4D55kir9QLH8rEYAaLXuv3c3sm8jYhY\nntor-onion-key Y/XgaHcPIJVa4D55kir9QLH8rEYAaLXuv3c3sm8jYhY\n");
    }

    @Test(expected=DescriptorParseException.class)
    public void testUnrecognizedLineFail() throws DescriptorParseException {
        String unrecognizedLine = "unrecognized-line 1";
        DescriptorBuilder.createWithUnrecognizedLine(unrecognizedLine, true);
    }

    @Test
    public void testUnrecognizedLineIgnore() throws DescriptorParseException {
        String unrecognizedLine = "unrecognized-line 1";
        ServerDescriptor descriptor = DescriptorBuilder.createWithUnrecognizedLine(unrecognizedLine, false);
        ArrayList<String> unrecognizedLines = new ArrayList<String>();
        unrecognizedLines.add(unrecognizedLine);
        Assert.assertEquals(unrecognizedLines, descriptor.getUnrecognizedLines());
    }

    @Test
    public void testSomeOtherKey() throws DescriptorParseException {
        ArrayList<String> unrecognizedLines = new ArrayList<String>();
        unrecognizedLines.add("some-other-key");
        unrecognizedLines.add("-----BEGIN RSA PUBLIC KEY-----");
        unrecognizedLines.add("MIGJAoGBAKM+iiHhO6eHsvd6Xjws9z9EQB1V/Bpuy5ciGJ1U4V9SeiKooSo5BpPL");
        unrecognizedLines.add("o3XT+6PIgzl3R6uycjS3Ejk47vLEJdcVTm/VG6E0ppu3olIynCI4QryfCEuC3cTF");
        unrecognizedLines.add("9wE4WXY4nX7w0RTN18UVLxrt1A9PP0cobFNiPs9rzJCbKFfacOkpAgMBAAE=");
        unrecognizedLines.add("-----END RSA PUBLIC KEY-----");
        StringBuilder sb = new StringBuilder();
        for (String line : unrecognizedLines) {
            sb.append("\n").append(line);
        }
        ServerDescriptor descriptor = DescriptorBuilder.createWithUnrecognizedLine(sb.toString().substring(1), false);
        Assert.assertEquals(unrecognizedLines, descriptor.getUnrecognizedLines());
    }

    @Test
    public void testUnrecognizedCryptoBlockNoKeyword() throws DescriptorParseException {
        ArrayList<String> unrecognizedLines = new ArrayList<String>();
        unrecognizedLines.add("-----BEGIN RSA PUBLIC KEY-----");
        unrecognizedLines.add("MIGJAoGBAKM+iiHhO6eHsvd6Xjws9z9EQB1V/Bpuy5ciGJ1U4V9SeiKooSo5BpPL");
        unrecognizedLines.add("o3XT+6PIgzl3R6uycjS3Ejk47vLEJdcVTm/VG6E0ppu3olIynCI4QryfCEuC3cTF");
        unrecognizedLines.add("9wE4WXY4nX7w0RTN18UVLxrt1A9PP0cobFNiPs9rzJCbKFfacOkpAgMBAAE=");
        unrecognizedLines.add("-----END RSA PUBLIC KEY-----");
        StringBuilder sb = new StringBuilder();
        for (String line : unrecognizedLines) {
            sb.append("\n").append(line);
        }
        ServerDescriptor descriptor = DescriptorBuilder.createWithUnrecognizedLine(sb.toString().substring(1), false);
        Assert.assertEquals(unrecognizedLines, descriptor.getUnrecognizedLines());
    }

    @Test
    public void testEd25519() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithEd25519Lines(IDENTITY_ED25519_LINES, MASTER_KEY_ED25519_LINE, ROUTER_SIG_ED25519_LINE);
        Assert.assertEquals((Object)IDENTITY_ED25519_LINES.substring(IDENTITY_ED25519_LINES.indexOf("\n") + 1), (Object)descriptor.getIdentityEd25519());
        Assert.assertEquals((Object)MASTER_KEY_ED25519_LINE.substring(MASTER_KEY_ED25519_LINE.indexOf(" ") + 1), (Object)descriptor.getMasterKeyEd25519());
        Assert.assertEquals((Object)ROUTER_SIG_ED25519_LINE.substring(ROUTER_SIG_ED25519_LINE.indexOf(" ") + 1), (Object)descriptor.getRouterSignatureEd25519());
    }

    @Test(expected=DescriptorParseException.class)
    public void testEd25519IdentityMasterKeyMismatch() throws DescriptorParseException {
        DescriptorBuilder.createWithEd25519Lines(IDENTITY_ED25519_LINES, "master-key-ed25519 AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA", ROUTER_SIG_ED25519_LINE);
    }

    @Test
    public void testEd25519IdentityMissing() throws DescriptorParseException {
        DescriptorBuilder.createWithEd25519Lines(null, MASTER_KEY_ED25519_LINE, ROUTER_SIG_ED25519_LINE);
    }

    @Test(expected=DescriptorParseException.class)
    public void testEd25519IdentityDuplicate() throws DescriptorParseException {
        DescriptorBuilder.createWithEd25519Lines("identity-ed25519\n-----BEGIN ED25519 CERT-----\nAQQABiX1AVGv5BuzJroQXbOh6vv1nbwc5rh2S13PyRFuLhTiifK4AQAgBACBCMwr\n4qgIlFDIzoC9ieJOtSkwrK+yXJPKlP8ojvgkx8cGKvhokOwA1eYDombzfwHcJ1EV\nbhEn/6g8i7wzO3LoqefIUrSAeEExOAOmm5mNmUIzL8EtnT6JHCr/sqUTUgA=\n-----END ED25519 CERT-----\nidentity-ed25519\n-----BEGIN ED25519 CERT-----\nAQQABiX1AVGv5BuzJroQXbOh6vv1nbwc5rh2S13PyRFuLhTiifK4AQAgBACBCMwr\n4qgIlFDIzoC9ieJOtSkwrK+yXJPKlP8ojvgkx8cGKvhokOwA1eYDombzfwHcJ1EV\nbhEn/6g8i7wzO3LoqefIUrSAeEExOAOmm5mNmUIzL8EtnT6JHCr/sqUTUgA=\n-----END ED25519 CERT-----", MASTER_KEY_ED25519_LINE, ROUTER_SIG_ED25519_LINE);
    }

    @Test(expected=DescriptorParseException.class)
    public void testEd25519IdentityEmptyCrypto() throws DescriptorParseException {
        DescriptorBuilder.createWithEd25519Lines("identity-ed25519\n-----BEGIN ED25519 CERT-----\n-----END ED25519 CERT-----", MASTER_KEY_ED25519_LINE, ROUTER_SIG_ED25519_LINE);
    }

    @Test
    public void testEd25519MasterKeyMissing() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithEd25519Lines(IDENTITY_ED25519_LINES, null, ROUTER_SIG_ED25519_LINE);
        Assert.assertEquals((Object)MASTER_KEY_ED25519_LINE.substring(MASTER_KEY_ED25519_LINE.indexOf(" ") + 1), (Object)descriptor.getMasterKeyEd25519());
    }

    @Test(expected=DescriptorParseException.class)
    public void testEd25519MasterKeyDuplicate() throws DescriptorParseException {
        DescriptorBuilder.createWithEd25519Lines(IDENTITY_ED25519_LINES, "master-key-ed25519 gQjMK+KoCJRQyM6AvYniTrUpMKyvslyTypT/KI74JMc\nmaster-key-ed25519 gQjMK+KoCJRQyM6AvYniTrUpMKyvslyTypT/KI74JMc", ROUTER_SIG_ED25519_LINE);
    }

    @Test
    public void testEd25519RouterSigMissing() throws DescriptorParseException {
        DescriptorBuilder.createWithEd25519Lines(IDENTITY_ED25519_LINES, MASTER_KEY_ED25519_LINE, null);
    }

    @Test(expected=DescriptorParseException.class)
    public void testEd25519RouterSigDuplicate() throws DescriptorParseException {
        DescriptorBuilder.createWithEd25519Lines(IDENTITY_ED25519_LINES, MASTER_KEY_ED25519_LINE, "router-sig-ed25519 y7WF9T2GFwkSDPZEhB55HgquIFOl5uXUFMYJPq3CXXUTKeJkSrtaZUB5s34fWdHQNtl84mH4dVaFMunHnwgYAw\nrouter-sig-ed25519 y7WF9T2GFwkSDPZEhB55HgquIFOl5uXUFMYJPq3CXXUTKeJkSrtaZUB5s34fWdHQNtl84mH4dVaFMunHnwgYAw");
    }

    @Test
    public void testOnionKeyCrosscert() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithOnionKeyCrosscertLines(ONION_KEY_CROSSCERT_LINES);
        Assert.assertEquals((Object)ONION_KEY_CROSSCERT_LINES.substring(ONION_KEY_CROSSCERT_LINES.indexOf("\n") + 1), (Object)descriptor.getOnionKeyCrosscert());
    }

    @Test(expected=DescriptorParseException.class)
    public void testOnionKeyCrosscertDuplicate() throws DescriptorParseException {
        DescriptorBuilder.createWithOnionKeyCrosscertLines("onion-key-crosscert\n-----BEGIN CROSSCERT-----\ngVWpiNgG2FekW1uonr4KKoqykjr4bqUBKGZfu6s9rvsV1TThnquZNP6ZhX2IPdQA\nlfKtzFggGu/4BiJ5oTSDj2sK2DMjY3rjrMQZ3I/wJ25yhc9gxjqYqUYO9MmJwALp\nfYkqp/t4WchJpyva/4hK8vITsI6eT2BfY/DWMy/suIE=\n-----END CROSSCERT-----\nonion-key-crosscert\n-----BEGIN CROSSCERT-----\ngVWpiNgG2FekW1uonr4KKoqykjr4bqUBKGZfu6s9rvsV1TThnquZNP6ZhX2IPdQA\nlfKtzFggGu/4BiJ5oTSDj2sK2DMjY3rjrMQZ3I/wJ25yhc9gxjqYqUYO9MmJwALp\nfYkqp/t4WchJpyva/4hK8vITsI6eT2BfY/DWMy/suIE=\n-----END CROSSCERT-----");
    }

    @Test
    public void testNtorOnionKeyCrosscert() throws DescriptorParseException {
        ServerDescriptor descriptor = DescriptorBuilder.createWithNtorOnionKeyCrosscertLines(NTOR_ONION_KEY_CROSSCERT_LINES);
        Assert.assertEquals((Object)NTOR_ONION_KEY_CROSSCERT_LINES.substring(NTOR_ONION_KEY_CROSSCERT_LINES.indexOf("\n") + 1), (Object)descriptor.getNtorOnionKeyCrosscert());
        Assert.assertEquals((long)1L, (long)descriptor.getNtorOnionKeyCrosscertSign());
    }

    @Test(expected=DescriptorParseException.class)
    public void testNtorOnionKeyCrosscertDuplicate() throws DescriptorParseException {
        DescriptorBuilder.createWithOnionKeyCrosscertLines("ntor-onion-key-crosscert 1\n-----BEGIN ED25519 CERT-----\nAQoABiUeAdauu1MxYGMmGLTCPaoes0RvW7udeLc1t8LZ4P3CDo5bAN4nrRfbCfOt\nz2Nwqn8tER1a+Ry6Vs+ilMZA55Rag4+f6Zdb1fmHWknCxbQlLHpqHACMtemPdaKa\nErPtMuiEqAc=\n-----END ED25519 CERT-----\nntor-onion-key-crosscert 1\n-----BEGIN ED25519 CERT-----\nAQoABiUeAdauu1MxYGMmGLTCPaoes0RvW7udeLc1t8LZ4P3CDo5bAN4nrRfbCfOt\nz2Nwqn8tER1a+Ry6Vs+ilMZA55Rag4+f6Zdb1fmHWknCxbQlLHpqHACMtemPdaKa\nErPtMuiEqAc=\n-----END ED25519 CERT-----");
    }

    private static class DescriptorBuilder {
        private String routerLine = "router saberrider2008 94.134.192.243 9001 0 0";
        private String bandwidthLine = "bandwidth 51200 51200 53470";
        private String platformLine = "platform Tor 0.2.2.35 (git-b04388f9e7546a9f) on Linux i686";
        private String publishedLine = "published 2012-01-01 04:03:19";
        private String fingerprintLine = "opt fingerprint D873 3048 FC8E C910 2466 AD8F 3098 622B F1BF 71FD";
        private String hibernatingLine = null;
        private String uptimeLine = "uptime 48";
        private String onionKeyLines = "onion-key\n-----BEGIN RSA PUBLIC KEY-----\nMIGJAoGBAKM+iiHhO6eHsvd6Xjws9z9EQB1V/Bpuy5ciGJ1U4V9SeiKooSo5BpPL\no3XT+6PIgzl3R6uycjS3Ejk47vLEJdcVTm/VG6E0ppu3olIynCI4QryfCEuC3cTF\n9wE4WXY4nX7w0RTN18UVLxrt1A9PP0cobFNiPs9rzJCbKFfacOkpAgMBAAE=\n-----END RSA PUBLIC KEY-----";
        private String signingKeyLines = "signing-key\n-----BEGIN RSA PUBLIC KEY-----\nMIGJAoGBALMm3r3QDh482Ewe6Ub9wvRIfmEkoNX6q5cEAtQRNHSDcNx41gjELbcl\nEniVMParBYACKfOxkS+mTTnIRDKVNEJTsDOwryNrc4X9JnPc/nn6ymYPiNDhUROG\n8URDIhQoixcUeyyrVB8sxliSstKimulGnB7xpjYOlO8JKaHLNL4TAgMBAAE=\n-----END RSA PUBLIC KEY-----";
        private String onionKeyCrosscertLines = null;
        private String ntorOnionKeyCrosscertLines = null;
        private String exitPolicyLines = "reject *:*";
        private String contactLine = "contact Random Person <nobody AT example dot com>";
        private String familyLine = null;
        private String readHistoryLine = null;
        private String writeHistoryLine = null;
        private String eventdnsLine = null;
        private String cachesExtraInfoLine = null;
        private String extraInfoDigestLine = "opt extra-info-digest 1469D1550738A25B1E7B47CDDBCD7B2899F51B74";
        private String hiddenServiceDirLine = "opt hidden-service-dir";
        private String protocolsLine = "opt protocols Link 1 2 Circuit 1";
        private String allowSingleHopExitsLine = null;
        private String ipv6PolicyLine = null;
        private String ntorOnionKeyLine = null;
        private String routerSignatureLines = "router-signature\n-----BEGIN SIGNATURE-----\no4j+kH8UQfjBwepUnr99v0ebN8RpzHJ/lqYsTojXHy9kMr1RNI9IDeSzA7PSqTuV\n4PL8QsGtlfwthtIoZpB2srZeyN/mcpA9fa1JXUrt/UN9K/+32Cyaad7h0nHE6Xfb\njqpXDpnBpvk4zjmzjjKYnIsUWTnADmu0fo3xTRqXi7g=\n-----END SIGNATURE-----";
        private String unrecognizedLine = null;
        private byte[] nonAsciiLineBytes = null;
        private String identityEd25519Lines = null;
        private String masterKeyEd25519Line = null;
        private String routerSigEd25519Line = null;

        private DescriptorBuilder() {
        }

        private static ServerDescriptor createWithRouterLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.routerLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithBandwidthLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.bandwidthLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithPlatformLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.platformLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithPublishedLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.publishedLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithFingerprintLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.fingerprintLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithHibernatingLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.hibernatingLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithUptimeLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.uptimeLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithOnionKeyLines(String lines) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.onionKeyLines = lines;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithSigningKeyLines(String lines) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.signingKeyLines = lines;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithOnionKeyCrosscertLines(String lines) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.onionKeyCrosscertLines = lines;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithNtorOnionKeyCrosscertLines(String lines) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.ntorOnionKeyCrosscertLines = lines;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithExitPolicyLines(String lines) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.exitPolicyLines = lines;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithContactLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.contactLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithFamilyLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.familyLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithReadHistoryLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.readHistoryLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithWriteHistoryLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.writeHistoryLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithEventdnsLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.eventdnsLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithCachesExtraInfoLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.cachesExtraInfoLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithExtraInfoDigestLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.extraInfoDigestLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithHiddenServiceDirLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.hiddenServiceDirLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithProtocolsLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.protocolsLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithAllowSingleHopExitsLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.allowSingleHopExitsLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithIpv6PolicyLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.ipv6PolicyLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithNtorOnionKeyLine(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.ntorOnionKeyLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithRouterSignatureLines(String line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.routerSignatureLines = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private static ServerDescriptor createWithUnrecognizedLine(String line, boolean failUnrecognizedDescriptorLines) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.unrecognizedLine = line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), failUnrecognizedDescriptorLines);
        }

        private static ServerDescriptor createWithNonAsciiLineBytes(byte[] lineBytes, boolean failUnrecognizedDescriptorLines) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.nonAsciiLineBytes = lineBytes;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), failUnrecognizedDescriptorLines);
        }

        private static ServerDescriptor createWithEd25519Lines(String identityEd25519Lines, String masterKeyEd25519Line, String routerSigEd25519Line) throws DescriptorParseException {
            DescriptorBuilder db = new DescriptorBuilder();
            db.identityEd25519Lines = identityEd25519Lines;
            db.masterKeyEd25519Line = masterKeyEd25519Line;
            db.routerSigEd25519Line = routerSigEd25519Line;
            return new RelayServerDescriptorImpl(db.buildDescriptor(), true);
        }

        private byte[] buildDescriptor() {
            StringBuilder sb = new StringBuilder();
            if (this.routerLine != null) {
                sb.append(this.routerLine).append("\n");
            }
            if (this.identityEd25519Lines != null) {
                sb.append(this.identityEd25519Lines).append("\n");
            }
            if (this.masterKeyEd25519Line != null) {
                sb.append(this.masterKeyEd25519Line).append("\n");
            }
            if (this.bandwidthLine != null) {
                sb.append(this.bandwidthLine).append("\n");
            }
            if (this.platformLine != null) {
                sb.append(this.platformLine).append("\n");
            }
            if (this.publishedLine != null) {
                sb.append(this.publishedLine).append("\n");
            }
            if (this.fingerprintLine != null) {
                sb.append(this.fingerprintLine).append("\n");
            }
            if (this.hibernatingLine != null) {
                sb.append(this.hibernatingLine).append("\n");
            }
            if (this.uptimeLine != null) {
                sb.append(this.uptimeLine).append("\n");
            }
            if (this.onionKeyLines != null) {
                sb.append(this.onionKeyLines).append("\n");
            }
            if (this.signingKeyLines != null) {
                sb.append(this.signingKeyLines).append("\n");
            }
            if (this.onionKeyCrosscertLines != null) {
                sb.append(this.onionKeyCrosscertLines).append("\n");
            }
            if (this.ntorOnionKeyCrosscertLines != null) {
                sb.append(this.ntorOnionKeyCrosscertLines).append("\n");
            }
            if (this.exitPolicyLines != null) {
                sb.append(this.exitPolicyLines).append("\n");
            }
            if (this.contactLine != null) {
                sb.append(this.contactLine).append("\n");
            }
            if (this.familyLine != null) {
                sb.append(this.familyLine).append("\n");
            }
            if (this.readHistoryLine != null) {
                sb.append(this.readHistoryLine).append("\n");
            }
            if (this.writeHistoryLine != null) {
                sb.append(this.writeHistoryLine).append("\n");
            }
            if (this.eventdnsLine != null) {
                sb.append(this.eventdnsLine).append("\n");
            }
            if (this.cachesExtraInfoLine != null) {
                sb.append(this.cachesExtraInfoLine).append("\n");
            }
            if (this.extraInfoDigestLine != null) {
                sb.append(this.extraInfoDigestLine).append("\n");
            }
            if (this.hiddenServiceDirLine != null) {
                sb.append(this.hiddenServiceDirLine).append("\n");
            }
            if (this.protocolsLine != null) {
                sb.append(this.protocolsLine).append("\n");
            }
            if (this.allowSingleHopExitsLine != null) {
                sb.append(this.allowSingleHopExitsLine).append("\n");
            }
            if (this.ipv6PolicyLine != null) {
                sb.append(this.ipv6PolicyLine).append("\n");
            }
            if (this.ntorOnionKeyLine != null) {
                sb.append(this.ntorOnionKeyLine).append("\n");
            }
            if (this.unrecognizedLine != null) {
                sb.append(this.unrecognizedLine).append("\n");
            }
            if (this.nonAsciiLineBytes != null) {
                try {
                    ByteArrayOutputStream baos = new ByteArrayOutputStream();
                    baos.write(sb.toString().getBytes());
                    baos.write(this.nonAsciiLineBytes);
                    baos.write("\n".getBytes());
                    if (this.routerSignatureLines != null) {
                        baos.write(this.routerSignatureLines.getBytes());
                    }
                    return baos.toByteArray();
                }
                catch (IOException e) {
                    return null;
                }
            }
            if (this.routerSigEd25519Line != null) {
                sb.append(this.routerSigEd25519Line).append("\n");
            }
            if (this.routerSignatureLines != null) {
                sb.append(this.routerSignatureLines).append("\n");
            }
            return sb.toString().getBytes();
        }
    }
}

