/*
 * Decompiled with CFR 0.152.
 */
package org.torproject.metrics.collector.relaydescs;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Locale;
import java.util.SortedSet;
import java.util.TimeZone;
import java.util.TreeSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.torproject.descriptor.Descriptor;
import org.torproject.descriptor.DescriptorReader;
import org.torproject.descriptor.DescriptorSourceFactory;
import org.torproject.descriptor.DirSourceEntry;
import org.torproject.descriptor.ExtraInfoDescriptor;
import org.torproject.descriptor.Microdescriptor;
import org.torproject.descriptor.NetworkStatusEntry;
import org.torproject.descriptor.RelayNetworkStatusConsensus;
import org.torproject.descriptor.RelayNetworkStatusVote;
import org.torproject.descriptor.ServerDescriptor;

public class ReferenceChecker {
    private static final Logger logger = LoggerFactory.getLogger(ReferenceChecker.class);
    private File descriptorsDir;
    private File referencesFile;
    private File historyFile;
    private long currentTimeMillis;
    private SortedSet<Reference> references = new TreeSet<Reference>();
    private static ObjectMapper objectMapper = new ObjectMapper().setVisibility(PropertyAccessor.ALL, JsonAutoDetect.Visibility.NONE).setVisibility(PropertyAccessor.FIELD, JsonAutoDetect.Visibility.ANY);
    private static DateFormat dateTimeFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'", Locale.US);
    private static final long ONE_HOUR = 3600000L;
    private static final long THREE_HOURS = 10800000L;
    private static final long SIX_HOURS = 21600000L;
    private static final long ONE_DAY = 86400000L;
    private static final long THIRTY_DAYS = 2592000000L;

    public ReferenceChecker(File descriptorsDir, File referencesFile, File historyFile) {
        this.descriptorsDir = descriptorsDir;
        this.referencesFile = referencesFile;
        this.historyFile = historyFile;
    }

    public void check() {
        this.getCurrentTimeMillis();
        this.readReferencesFile();
        this.readNewDescriptors();
        this.dropStaleReferences();
        this.checkReferences();
        this.writeReferencesFile();
    }

    private void getCurrentTimeMillis() {
        this.currentTimeMillis = System.currentTimeMillis();
    }

    private void readReferencesFile() {
        if (!this.referencesFile.exists()) {
            return;
        }
        try {
            this.references.addAll(Arrays.asList((Object[])objectMapper.readValue(this.referencesFile, Reference[].class)));
        }
        catch (IOException e) {
            logger.warn("Cannot read existing references file from previous run.", e);
        }
        catch (RuntimeException jpe) {
            logger.warn("Content of {} cannot be parsed. File will be erased and rewritten. In general, {} shouldn't be edited manually.  Error reason: {}", this.referencesFile.toString(), this.referencesFile.toString(), jpe.getMessage());
            try {
                Files.deleteIfExists(this.referencesFile.toPath());
            }
            catch (IOException ioe) {
                logger.warn("Cannot delete '{}', reason: {}", this.referencesFile.toString(), ioe.getMessage(), ioe);
            }
        }
    }

    private void readNewDescriptors() {
        DescriptorReader descriptorReader = DescriptorSourceFactory.createDescriptorReader();
        descriptorReader.setHistoryFile(this.historyFile);
        for (Descriptor descriptor : descriptorReader.readDescriptors(this.descriptorsDir)) {
            if (descriptor instanceof RelayNetworkStatusConsensus) {
                RelayNetworkStatusConsensus consensus = (RelayNetworkStatusConsensus)descriptor;
                String consensusFlavor = consensus.getConsensusFlavor();
                if (consensusFlavor == null) {
                    this.readRelayNetworkStatusConsensusUnflavored(consensus);
                    continue;
                }
                if (!consensusFlavor.equals("microdesc")) continue;
                this.readRelayNetworkStatusConsensusMicrodesc(consensus);
                continue;
            }
            if (descriptor instanceof RelayNetworkStatusVote) {
                this.readRelayNetworkStatusVote((RelayNetworkStatusVote)descriptor);
                continue;
            }
            if (descriptor instanceof ServerDescriptor) {
                this.readServerDescriptor((ServerDescriptor)descriptor);
                continue;
            }
            if (descriptor instanceof ExtraInfoDescriptor) {
                this.readExtraInfoDescriptor((ExtraInfoDescriptor)descriptor);
                continue;
            }
            if (!(descriptor instanceof Microdescriptor)) continue;
            this.readMicrodescriptor((Microdescriptor)descriptor);
        }
        descriptorReader.saveHistoryFile(this.historyFile);
    }

    private void readRelayNetworkStatusConsensusUnflavored(RelayNetworkStatusConsensus consensus) {
        String validAfter = dateTimeFormat.format(consensus.getValidAfterMillis());
        String referencing = String.format("C-%s", validAfter);
        this.addReference(referencing, String.format("M-%s", validAfter), 1.0, consensus.getValidAfterMillis() + 10800000L);
        for (DirSourceEntry dirSourceEntry : consensus.getDirSourceEntries().values()) {
            if (dirSourceEntry.isLegacy()) continue;
            this.addReference(referencing, String.format("V-%s-%s", validAfter, dirSourceEntry.getIdentity()), 1.0, consensus.getValidAfterMillis() + 10800000L);
        }
        double entryWeight = 200.0 / (double)consensus.getStatusEntries().size();
        for (NetworkStatusEntry entry : consensus.getStatusEntries().values()) {
            this.addReference(referencing, String.format("S-%s", entry.getDescriptor()), entryWeight, entry.getPublishedMillis() + 10800000L);
        }
    }

    private void readRelayNetworkStatusConsensusMicrodesc(RelayNetworkStatusConsensus consensus) {
        String validAfter = dateTimeFormat.format(consensus.getValidAfterMillis());
        String referencing = String.format("M-%s", validAfter);
        this.addReference(referencing, String.format("C-%s", validAfter), 1.0, consensus.getValidAfterMillis() + 10800000L);
        double entryWeight = 200.0 / (double)consensus.getStatusEntries().size();
        for (NetworkStatusEntry entry : consensus.getStatusEntries().values()) {
            for (String digest : entry.getMicrodescriptorDigestsSha256Base64()) {
                this.addReference(referencing, String.format("D-%s", digest), entryWeight, entry.getPublishedMillis() + 10800000L);
            }
        }
    }

    private void readRelayNetworkStatusVote(RelayNetworkStatusVote vote) {
        String validAfter = dateTimeFormat.format(vote.getValidAfterMillis());
        String referencing = String.format("V-%s-%s", validAfter, vote.getIdentity());
        double entryWeight = 200.0 / (double)vote.getStatusEntries().size();
        for (NetworkStatusEntry entry : vote.getStatusEntries().values()) {
            this.addReference(referencing, String.format("S-%s", entry.getDescriptor()), entryWeight, entry.getPublishedMillis() + 21600000L);
        }
    }

    private void readServerDescriptor(ServerDescriptor serverDescriptor) {
        String referenced = serverDescriptor.getExtraInfoDigestSha1Hex() == null ? "" : String.format("E-%s", serverDescriptor.getExtraInfoDigestSha1Hex());
        this.addReference(String.format("S-%s", serverDescriptor.getDigestSha1Hex()), referenced, 0.01, serverDescriptor.getPublishedMillis() + 21600000L);
    }

    private void readExtraInfoDescriptor(ExtraInfoDescriptor extraInfoDescriptor) {
        this.addReference(String.format("E-%s", extraInfoDescriptor.getDigestSha1Hex()), "", 0.005, extraInfoDescriptor.getPublishedMillis() + 21600000L);
    }

    private void readMicrodescriptor(Microdescriptor microdesc) {
        this.addReference(String.format("D-%s", microdesc.getDigestSha256Base64()), "", 0.0, this.currentTimeMillis + 2592000000L);
    }

    private void addReference(String referencing, String referenced, double weight, long expiresAfterMillis) {
        this.references.add(new Reference(referencing.toUpperCase(), referenced.toUpperCase(), weight, expiresAfterMillis));
    }

    private void dropStaleReferences() {
        TreeSet<Reference> recentReferences = new TreeSet<Reference>();
        for (Reference reference : this.references) {
            if (this.currentTimeMillis > reference.expiresAfterMillis) continue;
            recentReferences.add(reference);
        }
        this.references = recentReferences;
    }

    private void checkReferences() {
        HashSet<String> knownDescriptors = new HashSet<String>();
        for (Reference reference : this.references) {
            knownDescriptors.add(reference.referencing);
        }
        double totalMissingDescriptorsWeight = 0.0;
        TreeSet<String> missingDescriptors = new TreeSet<String>();
        StringBuilder sb = new StringBuilder("Missing referenced descriptors:");
        for (Reference reference : this.references) {
            if (reference.referenced.length() <= 0 || knownDescriptors.contains(reference.referenced)) continue;
            if (!missingDescriptors.contains(reference.referenced)) {
                totalMissingDescriptorsWeight += reference.weight;
            }
            missingDescriptors.add(reference.referenced);
            sb.append(String.format("%n%s -> %s (%.4f -> %.4f)", reference.referencing, reference.referenced, reference.weight, totalMissingDescriptorsWeight));
        }
        logger.info(sb.toString());
        if (totalMissingDescriptorsWeight > 0.999) {
            logger.warn("Missing too many referenced descriptors ({}).", (Object)totalMissingDescriptorsWeight);
        }
    }

    private void writeReferencesFile() {
        try {
            objectMapper.writeValue(this.referencesFile, this.references);
        }
        catch (IOException e) {
            logger.warn("Cannot write references file for next run.", e);
        }
    }

    static {
        dateTimeFormat.setLenient(false);
        dateTimeFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
    }

    private static class Reference
    implements Comparable<Reference> {
        private String referencing;
        private String referenced;
        private double weight;
        private long expiresAfterMillis;

        public Reference() {
        }

        public Reference(String referencing, String referenced, double weight, long expiresAfterMillis) {
            this.referencing = referencing;
            this.referenced = referenced;
            this.weight = weight;
            this.expiresAfterMillis = expiresAfterMillis;
        }

        public boolean equals(Object otherObject) {
            if (!(otherObject instanceof Reference)) {
                return false;
            }
            Reference other = (Reference)otherObject;
            return this.referencing.equals(other.referencing) && this.referenced.equals(other.referenced);
        }

        public int hashCode() {
            return this.referencing.hashCode() + this.referenced.hashCode();
        }

        @Override
        public int compareTo(Reference other) {
            int result = this.referencing.compareTo(other.referencing);
            if (result == 0) {
                result = this.referenced.compareTo(other.referenced);
            }
            return result;
        }
    }
}

