/*
 * Decompiled with CFR 0.152.
 */
package org.torproject.metrics.collector.onionperf;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.StandardCopyOption;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.compress.utils.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.torproject.descriptor.Descriptor;
import org.torproject.descriptor.DescriptorParser;
import org.torproject.descriptor.DescriptorSourceFactory;
import org.torproject.descriptor.TorperfResult;
import org.torproject.metrics.collector.conf.Configuration;
import org.torproject.metrics.collector.conf.ConfigurationException;
import org.torproject.metrics.collector.conf.Key;
import org.torproject.metrics.collector.cron.CollecTorMain;
import org.torproject.metrics.collector.downloader.Downloader;
import org.torproject.metrics.collector.persist.PersistenceUtils;

public class OnionPerfDownloader
extends CollecTorMain {
    private static final Logger logger = LoggerFactory.getLogger(OnionPerfDownloader.class);
    private static final String TORPERF = "torperf";
    private static final String ONIONPERF = "onionperf";
    private File onionPerfDownloadedFile;
    private SortedSet<String> downloadedFiles = new TreeSet<String>();
    private URL[] onionPerfHosts = null;
    private Map<URL, List<String>> tpfFileUrls = new HashMap<URL, List<String>>();
    private Map<URL, List<String>> onionPerfAnalysisFileUrls = new HashMap<URL, List<String>>();
    private File archiveDirectory = null;
    private File recentDirectory = null;
    private static final Pattern TPF_FILE_URL_PATTERN = Pattern.compile(".*<a href=\"([^\"]+\\.tpf)\">.*");
    private static final Pattern ONIONPERF_ANALYSIS_FILE_URL_PATTERN = Pattern.compile(".*<a href=\"([0-9-]{10}\\.onionperf\\.analysis\\.json\\.xz)\">.*");
    private static final DateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd");

    public OnionPerfDownloader(Configuration config) {
        super(config);
        this.mapPathDescriptors.put("recent/torperf", TorperfResult.class);
        this.mapPathDescriptors.put("recent/onionperf", TorperfResult.class);
    }

    @Override
    public String module() {
        return ONIONPERF;
    }

    @Override
    protected String syncMarker() {
        return "OnionPerf";
    }

    @Override
    protected void startProcessing() throws ConfigurationException {
        this.onionPerfDownloadedFile = new File(this.config.getPath(Key.StatsPath).toFile(), "onionperf-downloaded");
        this.onionPerfHosts = this.config.getUrlArray(Key.OnionPerfHosts);
        this.readDownloadedOnionPerfFiles();
        this.archiveDirectory = this.config.getPath(Key.OutputPath).toFile();
        this.recentDirectory = this.config.getPath(Key.RecentPath).toFile();
        for (URL baseUrl : this.onionPerfHosts) {
            this.downloadFromOnionPerfHost(baseUrl);
        }
        this.writeDownloadedOnionPerfFiles();
        this.cleanUpDirectories();
    }

    private void readDownloadedOnionPerfFiles() {
        if (!this.onionPerfDownloadedFile.exists()) {
            return;
        }
        try (BufferedReader br = new BufferedReader(new FileReader(this.onionPerfDownloadedFile));){
            String line;
            while ((line = br.readLine()) != null) {
                this.downloadedFiles.add(line);
            }
        }
        catch (IOException e) {
            logger.info("Unable to read download history file '{}'. Ignoring download history and downloading all available files.", (Object)this.onionPerfDownloadedFile.getAbsolutePath());
            this.downloadedFiles.clear();
        }
    }

    private void downloadFromOnionPerfHost(URL baseUrl) {
        logger.info("Downloading from OnionPerf host {}", (Object)baseUrl);
        this.downloadOnionPerfDirectoryListing(baseUrl);
        String source = baseUrl.getHost().split("\\.")[0];
        if (this.tpfFileUrls.containsKey(baseUrl)) {
            for (String tpfFileName : this.tpfFileUrls.get(baseUrl)) {
                this.downloadAndParseOnionPerfTpfFile(baseUrl, source, tpfFileName);
            }
        }
        if (this.onionPerfAnalysisFileUrls.containsKey(baseUrl)) {
            for (String onionPerfAnalysisFileName : this.onionPerfAnalysisFileUrls.get(baseUrl)) {
                this.downloadAndParseOnionPerfAnalysisFile(baseUrl, source, onionPerfAnalysisFileName);
            }
        }
    }

    private void downloadOnionPerfDirectoryListing(URL baseUrl) {
        try (BufferedReader br = new BufferedReader(new InputStreamReader(baseUrl.openStream()));){
            String line;
            while ((line = br.readLine()) != null) {
                Matcher onionPerfAnalysisFileMatcher;
                Matcher tpfFileMatcher = TPF_FILE_URL_PATTERN.matcher(line);
                if (tpfFileMatcher.matches() && !tpfFileMatcher.group(1).startsWith("/")) {
                    this.tpfFileUrls.putIfAbsent(baseUrl, new ArrayList());
                    this.tpfFileUrls.get(baseUrl).add(tpfFileMatcher.group(1));
                }
                if (!(onionPerfAnalysisFileMatcher = ONIONPERF_ANALYSIS_FILE_URL_PATTERN.matcher(line)).matches() || onionPerfAnalysisFileMatcher.group(1).startsWith("/")) continue;
                this.onionPerfAnalysisFileUrls.putIfAbsent(baseUrl, new ArrayList());
                this.onionPerfAnalysisFileUrls.get(baseUrl).add(onionPerfAnalysisFileMatcher.group(1));
            }
        }
        catch (IOException e) {
            logger.warn("Unable to download directory listing from '{}'.  Skipping this OnionPerf host.", (Object)baseUrl);
            this.tpfFileUrls.remove(baseUrl);
            this.onionPerfAnalysisFileUrls.remove(baseUrl);
        }
    }

    private void downloadAndParseOnionPerfTpfFile(URL baseUrl, String source, String tpfFileName) {
        byte[] rawDescriptorBytes;
        byte[] downloadedBytes;
        String date;
        int fileSize;
        URL tpfFileUrl;
        try {
            tpfFileUrl = new URL(baseUrl, tpfFileName);
        }
        catch (MalformedURLException e1) {
            logger.warn("Unable to put together base URL '{}' and .tpf file path '{}' to a URL.  Skipping.", (Object)baseUrl, (Object)tpfFileName);
            return;
        }
        if (this.downloadedFiles.contains(tpfFileUrl.toString())) {
            return;
        }
        String[] tpfFileNameParts = tpfFileName.split("-");
        if (!tpfFileName.startsWith(source + "-") || tpfFileName.length() < "s-f-yyyy-MM-dd".length() || tpfFileNameParts.length < 5) {
            logger.warn("Invalid .tpf file name '{}{}'.  Skipping.", (Object)baseUrl, (Object)tpfFileName);
            return;
        }
        try {
            fileSize = Integer.parseInt(tpfFileNameParts[tpfFileNameParts.length - 4]);
            date = tpfFileName.substring(tpfFileName.length() - 14, tpfFileName.length() - 4);
            DATE_FORMAT.parse(date);
        }
        catch (NumberFormatException | ParseException e) {
            logger.warn("Invalid .tpf file name '{}{}'.  Skipping.", new Object[]{baseUrl, tpfFileName, e});
            return;
        }
        File tempFile = new File(this.recentDirectory, "torperf/." + tpfFileName);
        try {
            downloadedBytes = Downloader.downloadFromHttpServer(new URL(baseUrl + tpfFileName));
        }
        catch (IOException e) {
            logger.warn("Unable to download '{}{}'. Skipping.", new Object[]{baseUrl, tpfFileName, e});
            return;
        }
        if (null == downloadedBytes) {
            logger.warn("Unable to download '{}{}'. Skipping.", (Object)baseUrl, (Object)tpfFileName);
            return;
        }
        tempFile.getParentFile().mkdirs();
        try {
            Files.write(tempFile.toPath(), downloadedBytes, new OpenOption[0]);
        }
        catch (IOException e) {
            logger.warn("Unable to write previously downloaded '{}{}' to temporary file '{}'. Skipping.", new Object[]{baseUrl, tpfFileName, tempFile, e});
            return;
        }
        DescriptorParser descriptorParser = DescriptorSourceFactory.createDescriptorParser();
        try {
            rawDescriptorBytes = Files.readAllBytes(tempFile.toPath());
        }
        catch (IOException e) {
            logger.warn("OnionPerf file '{}{}' could not be read.  Skipping.", new Object[]{baseUrl, tpfFileName, e});
            tempFile.delete();
            return;
        }
        Iterable descriptors = descriptorParser.parseDescriptors(rawDescriptorBytes, null, tpfFileName);
        String message = null;
        for (Descriptor descriptor : descriptors) {
            if (!(descriptor instanceof TorperfResult)) {
                message = "File contains descriptors other than Torperf results.";
                break;
            }
            TorperfResult torperf = (TorperfResult)descriptor;
            if (!source.equals(torperf.getSource())) {
                message = "File contains Torperf result from another source.";
                break;
            }
            if (fileSize != torperf.getFileSize()) {
                message = "File contains Torperf result from another file size.";
                break;
            }
            if (date.equals(DATE_FORMAT.format(torperf.getStartMillis()))) continue;
            message = "File contains Torperf result from another date.";
            break;
        }
        if (null != message) {
            logger.warn("OnionPerf file '{}{}' was found to be invalid: {}.  Skipping.", new Object[]{baseUrl, tpfFileName, message});
            tempFile.delete();
            return;
        }
        File archiveFile = new File(this.archiveDirectory, "torperf/" + date.replaceAll("-", "/") + "/" + tpfFileName);
        archiveFile.getParentFile().mkdirs();
        try {
            Files.copy(tempFile.toPath(), archiveFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            logger.warn("Unable to copy OnionPerf file {} to {}.  Skipping.", new Object[]{tempFile, archiveFile, e});
            tempFile.delete();
            return;
        }
        File recentFile = new File(this.recentDirectory, "torperf/" + tpfFileName);
        tempFile.renameTo(recentFile);
        this.downloadedFiles.add(baseUrl + tpfFileName);
    }

    private void downloadAndParseOnionPerfAnalysisFile(URL baseUrl, String source, String onionPerfAnalysisFileName) {
        byte[] rawDescriptorBytes;
        byte[] downloadedBytes;
        String date;
        URL onionPerfAnalysisFileUrl;
        try {
            onionPerfAnalysisFileUrl = new URL(baseUrl, onionPerfAnalysisFileName);
        }
        catch (MalformedURLException e1) {
            logger.warn("Unable to put together base URL '{}' and file path '{}' to a URL. Skipping.", (Object)baseUrl, (Object)onionPerfAnalysisFileName);
            return;
        }
        if (this.downloadedFiles.contains(onionPerfAnalysisFileUrl.toString())) {
            return;
        }
        try {
            date = onionPerfAnalysisFileName.substring(0, 10);
            DATE_FORMAT.parse(date);
        }
        catch (NumberFormatException | ParseException e) {
            logger.warn("Invalid file name '{}{}'. Skipping.", new Object[]{baseUrl, onionPerfAnalysisFileName, e});
            return;
        }
        File tempFile = new File(this.recentDirectory, "onionperf/." + onionPerfAnalysisFileName);
        try {
            downloadedBytes = Downloader.downloadFromHttpServer(new URL(baseUrl + onionPerfAnalysisFileName));
        }
        catch (IOException e) {
            logger.warn("Unable to download '{}{}'. Skipping.", new Object[]{baseUrl, onionPerfAnalysisFileName, e});
            return;
        }
        if (null == downloadedBytes) {
            logger.warn("Unable to download '{}{}'. Skipping.", (Object)baseUrl, (Object)onionPerfAnalysisFileName);
            return;
        }
        tempFile.getParentFile().mkdirs();
        try {
            Files.write(tempFile.toPath(), downloadedBytes, new OpenOption[0]);
        }
        catch (IOException e) {
            logger.warn("Unable to write previously downloaded '{}{}' to temporary file '{}'. Skipping.", new Object[]{baseUrl, onionPerfAnalysisFileName, tempFile, e});
            return;
        }
        DescriptorParser descriptorParser = DescriptorSourceFactory.createDescriptorParser();
        try {
            rawDescriptorBytes = IOUtils.toByteArray((InputStream)Files.newInputStream(tempFile.toPath(), new OpenOption[0]));
        }
        catch (IOException e) {
            logger.warn("OnionPerf file '{}{}' could not be read. Skipping.", new Object[]{baseUrl, onionPerfAnalysisFileName, e});
            tempFile.delete();
            return;
        }
        Iterable descriptors = descriptorParser.parseDescriptors(rawDescriptorBytes, null, onionPerfAnalysisFileName);
        String message = null;
        for (Descriptor descriptor : descriptors) {
            if (!(descriptor instanceof TorperfResult)) {
                message = "File contains descriptors other than an OnionPerf analysis document: " + descriptor.getClass();
                break;
            }
            TorperfResult torperf = (TorperfResult)descriptor;
            if (source.equals(torperf.getSource())) continue;
            message = "File contains transfer from another source: " + torperf.getSource();
            break;
        }
        if (null != message) {
            logger.warn("OnionPerf file '{}{}' was found to be invalid: {}. Skipping.", new Object[]{baseUrl, onionPerfAnalysisFileName, message});
            tempFile.delete();
            return;
        }
        File archiveFile = new File(this.archiveDirectory, "onionperf/" + date.replaceAll("-", "/") + "/" + date + "." + source + ".onionperf.analysis.json.xz");
        archiveFile.getParentFile().mkdirs();
        try {
            Files.copy(tempFile.toPath(), archiveFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            logger.warn("Unable to copy OnionPerf file {} to {}. Skipping.", new Object[]{tempFile, archiveFile, e});
            tempFile.delete();
            return;
        }
        File recentFile = new File(this.recentDirectory, "onionperf/" + date + "." + source + ".onionperf.analysis.json.xz");
        tempFile.renameTo(recentFile);
        this.downloadedFiles.add(baseUrl + onionPerfAnalysisFileName);
    }

    private void writeDownloadedOnionPerfFiles() {
        this.onionPerfDownloadedFile.getParentFile().mkdirs();
        try (BufferedWriter bw = new BufferedWriter(new FileWriter(this.onionPerfDownloadedFile));){
            for (String line : this.downloadedFiles) {
                bw.write(line);
                bw.newLine();
            }
        }
        catch (IOException e) {
            logger.warn("Unable to write download history file '{}'.  This may result in ignoring history and downloading all available .tpf files in the next execution.", (Object)this.onionPerfDownloadedFile.getAbsolutePath(), (Object)e);
        }
    }

    private void cleanUpDirectories() {
        PersistenceUtils.cleanDirectory(new File(this.recentDirectory, TORPERF).toPath(), Instant.now().minus(3L, ChronoUnit.DAYS).toEpochMilli());
        PersistenceUtils.cleanDirectory(new File(this.recentDirectory, ONIONPERF).toPath(), Instant.now().minus(3L, ChronoUnit.DAYS).toEpochMilli());
        PersistenceUtils.cleanDirectory(new File(this.archiveDirectory, TORPERF).toPath(), Instant.now().minus(49L, ChronoUnit.DAYS).toEpochMilli());
        PersistenceUtils.cleanDirectory(new File(this.archiveDirectory, ONIONPERF).toPath(), Instant.now().minus(49L, ChronoUnit.DAYS).toEpochMilli());
    }

    static {
        DATE_FORMAT.setLenient(false);
    }
}

