/*
 * Decompiled with CFR 0.152.
 */
package org.torproject.metrics.collector.bridgepools;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.DateTimeException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.Stack;
import java.util.TreeMap;
import java.util.TreeSet;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.torproject.descriptor.BridgePoolAssignment;
import org.torproject.metrics.collector.conf.Configuration;
import org.torproject.metrics.collector.conf.ConfigurationException;
import org.torproject.metrics.collector.conf.Key;
import org.torproject.metrics.collector.cron.CollecTorMain;
import org.torproject.metrics.collector.persist.PersistenceUtils;

public class BridgePoolAssignmentsProcessor
extends CollecTorMain {
    private static final Logger logger = LoggerFactory.getLogger(BridgePoolAssignmentsProcessor.class);
    private File assignmentsDirectory;
    private Path parsedBridgePoolAssignmentsFile;
    private String outputPathName;
    private String recentPathName;
    private final DateTimeFormatter assignmentFormat = DateTimeFormatter.ofPattern("uuuu-MM-dd HH:mm:ss");
    private final DateTimeFormatter filenameFormat = DateTimeFormatter.ofPattern("uuuu/MM/dd/uuuu-MM-dd-HH-mm-ss");

    public BridgePoolAssignmentsProcessor(Configuration config) {
        super(config);
        this.mapPathDescriptors.put("recent/bridge-pool-assignments", BridgePoolAssignment.class);
    }

    @Override
    public String module() {
        return "BridgePoolAssignments";
    }

    @Override
    protected String syncMarker() {
        return "BridgePoolAssignments";
    }

    @Override
    protected void startProcessing() throws ConfigurationException {
        logger.info("Starting bridge-pool-assignments module of CollecTor.");
        this.initializeConfiguration();
        SortedSet<Path> previouslyProcessedFiles = this.readProcessedFiles(this.parsedBridgePoolAssignmentsFile);
        TreeSet<Path> processedFiles = new TreeSet<Path>();
        List<File> assignmentFiles = this.listAssignmentFiles();
        LocalDateTime latestPublished = null;
        for (File assignmentFile : assignmentFiles) {
            processedFiles.add(assignmentFile.toPath());
            if (previouslyProcessedFiles.contains(assignmentFile.toPath())) continue;
            logger.info("Processing bridge pool assignment file '{}'...", (Object)assignmentFile.getAbsolutePath());
            SortedMap<LocalDateTime, SortedMap<String, String>> readBridgePoolAssignments = this.readBridgePoolAssignments(assignmentFile);
            if (null == latestPublished || readBridgePoolAssignments.lastKey().isAfter(latestPublished)) {
                latestPublished = readBridgePoolAssignments.lastKey();
            }
            for (Map.Entry<LocalDateTime, SortedMap<String, String>> e : readBridgePoolAssignments.entrySet()) {
                File[] outputFiles;
                LocalDateTime published = e.getKey();
                SortedMap<String, String> originalAssignments = e.getValue();
                SortedMap<String, String> sanitizedAssignments = this.sanitizeAssignments(originalAssignments);
                if (null == sanitizedAssignments) {
                    logger.warn("Unable to sanitize assignments published at {}. Skipping.", (Object)published);
                    continue;
                }
                String formattedSanitizedAssignments = this.formatSanitizedAssignments(published, sanitizedAssignments);
                File tarballFile = Paths.get(this.outputPathName, published.format(this.filenameFormat)).toFile();
                File rsyncFile = new File(this.recentPathName, tarballFile.getName());
                for (File outputFile : outputFiles = new File[]{tarballFile, rsyncFile}) {
                    if (outputFile.exists()) continue;
                    this.writeSanitizedAssignmentsToFile(outputFile, formattedSanitizedAssignments);
                }
            }
        }
        if (null != latestPublished && latestPublished.plusMinutes(330L).isBefore(LocalDateTime.now())) {
            logger.warn("The last known bridge pool assignment list was published at {}, which is more than 5:30 hours in the past.", (Object)latestPublished);
        }
        this.writeProcessedFiles(this.parsedBridgePoolAssignmentsFile, processedFiles);
        this.cleanUpDirectories();
        logger.info("Finished processing bridge pool assignment file(s).");
    }

    private void initializeConfiguration() throws ConfigurationException {
        this.parsedBridgePoolAssignmentsFile = this.config.getPath(Key.StatsPath).resolve("parsed-bridge-pool-assignments");
        this.outputPathName = Paths.get(this.config.getPath(Key.OutputPath).toString(), "bridge-pool-assignments").toString();
        this.recentPathName = Paths.get(this.config.getPath(Key.RecentPath).toString(), "bridge-pool-assignments").toString();
        this.assignmentsDirectory = this.config.getPath(Key.BridgePoolAssignmentsLocalOrigins).toFile();
    }

    private List<File> listAssignmentFiles() {
        ArrayList<File> assignmentFiles = new ArrayList<File>();
        Stack<File> files = new Stack<File>();
        files.add(this.assignmentsDirectory);
        while (!files.isEmpty()) {
            File file = (File)files.pop();
            if (file.isDirectory()) {
                File[] filesInDirectory = file.listFiles();
                if (null == filesInDirectory) continue;
                files.addAll(Arrays.asList(filesInDirectory));
                continue;
            }
            if (!file.getName().startsWith("assignments.log")) continue;
            assignmentFiles.add(file);
        }
        return assignmentFiles;
    }

    private SortedMap<LocalDateTime, SortedMap<String, String>> readBridgePoolAssignments(File assignmentFile) {
        TreeMap<LocalDateTime, SortedMap<String, String>> readBridgePoolAssignments = new TreeMap<LocalDateTime, SortedMap<String, String>>();
        try {
            String line;
            BufferedReader br = assignmentFile.getName().endsWith(".gz") ? new BufferedReader(new InputStreamReader((InputStream)new GzipCompressorInputStream((InputStream)new FileInputStream(assignmentFile)))) : new BufferedReader(new FileReader(assignmentFile));
            TreeMap<String, String> currentAssignments = null;
            while ((line = br.readLine()) != null) {
                if (line.startsWith("bridge-pool-assignment ")) {
                    try {
                        LocalDateTime bridgePoolAssignmentTime = LocalDateTime.parse(line.substring("bridge-pool-assignment ".length()), this.assignmentFormat);
                        if (readBridgePoolAssignments.containsKey(bridgePoolAssignmentTime)) {
                            logger.warn("Input file {} contains duplicate line: {}. Discarding previously read line and subsequent assignment lines.", (Object)assignmentFile, (Object)line);
                        }
                        currentAssignments = new TreeMap<String, String>();
                        readBridgePoolAssignments.put(bridgePoolAssignmentTime, currentAssignments);
                        continue;
                    }
                    catch (DateTimeException e) {
                        logger.warn("Could not parse timestamp from line {}. Skipping bridge pool assignment file '{}'.", new Object[]{line, assignmentFile.getAbsolutePath(), e});
                        break;
                    }
                }
                if (null == currentAssignments) {
                    logger.warn("Input file {} does not start with a bridge-pool-assignments line. Skipping.", (Object)assignmentFile);
                    break;
                }
                String[] parts = line.split(" ", 2);
                if (parts.length < 2 || parts[0].length() < 40) {
                    logger.warn("Unrecognized line '{}'. Aborting.", (Object)line);
                    break;
                }
                if (currentAssignments.containsKey(parts[0])) {
                    logger.warn("Input file {} contains duplicate line: {}. Discarding previously read line.", (Object)assignmentFile, (Object)line);
                }
                currentAssignments.put(parts[0], parts[1]);
            }
            br.close();
        }
        catch (IOException e) {
            logger.warn("Could not read bridge pool assignment file '{}'. Skipping.", (Object)assignmentFile.getAbsolutePath(), (Object)e);
        }
        return readBridgePoolAssignments;
    }

    private SortedMap<String, String> sanitizeAssignments(SortedMap<String, String> originalAssignments) {
        TreeMap<String, String> sanitizedAssignments = new TreeMap<String, String>();
        for (Map.Entry<String, String> e : originalAssignments.entrySet()) {
            String originalFingerprint = e.getKey();
            String assignmentDetails = e.getValue();
            try {
                String hashedFingerprint = Hex.encodeHexString((byte[])DigestUtils.sha1((byte[])Hex.decodeHex((char[])originalFingerprint.toCharArray()))).toLowerCase();
                sanitizedAssignments.put(hashedFingerprint, assignmentDetails);
            }
            catch (DecoderException ex) {
                logger.warn("Unable to decode hex fingerprint. Aborting.", (Throwable)ex);
                return null;
            }
        }
        return sanitizedAssignments;
    }

    private String formatSanitizedAssignments(LocalDateTime published, SortedMap<String, String> sanitizedAssignments) {
        StringBuilder sb = new StringBuilder();
        sb.append("@type bridge-pool-assignment 1.0\n");
        sb.append(String.format("bridge-pool-assignment %s\n", published.format(this.assignmentFormat)));
        for (Map.Entry<String, String> e : sanitizedAssignments.entrySet()) {
            sb.append(String.format("%s %s%n", e.getKey(), e.getValue()));
        }
        return sb.toString();
    }

    private void writeSanitizedAssignmentsToFile(File outputFile, String formattedSanitizedAssignments) {
        if (!outputFile.getParentFile().exists() && !outputFile.getParentFile().mkdirs()) {
            logger.warn("Could not create parent directories of {}.", (Object)outputFile);
            return;
        }
        try (BufferedWriter bw = new BufferedWriter(new FileWriter(outputFile));){
            bw.write(formattedSanitizedAssignments);
        }
        catch (IOException e) {
            logger.warn("Unable to write sanitized bridge pool assignments to {}.", (Object)outputFile, (Object)e);
        }
    }

    public void cleanUpDirectories() {
        PersistenceUtils.cleanDirectory(Paths.get(this.recentPathName, new String[0]), Instant.now().minus(3L, ChronoUnit.DAYS).toEpochMilli());
        PersistenceUtils.cleanDirectory(Paths.get(this.outputPathName, new String[0]), Instant.now().minus(49L, ChronoUnit.DAYS).toEpochMilli());
    }
}

