/*
 * Decompiled with CFR 0.152.
 */
package org.torproject.metrics.collector.persist;

import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.util.Date;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PersistenceUtils {
    private static final Logger logger = LoggerFactory.getLogger(PersistenceUtils.class);
    public static final String TEMPFIX = ".tmp";

    public static boolean storeToFileSystem(byte[] typeAnnotation, byte[] data, Path outputPath, StandardOpenOption option) {
        return PersistenceUtils.storeToFileSystem(typeAnnotation, data, outputPath, option, false);
    }

    public static boolean storeToFileSystem(byte[] typeAnnotation, byte[] data, Path outputPath, StandardOpenOption option, boolean useTmp) {
        Path tmpPath = outputPath;
        try {
            if (useTmp) {
                tmpPath = new File(outputPath.toFile().getParent(), outputPath.toFile().getName() + TEMPFIX).toPath();
                if (Files.exists(outputPath, new LinkOption[0]) && StandardOpenOption.CREATE_NEW == option) {
                    return false;
                }
                if (Files.exists(outputPath, new LinkOption[0]) && !Files.exists(tmpPath, new LinkOption[0]) && StandardOpenOption.APPEND == option) {
                    Files.copy(outputPath, tmpPath, StandardCopyOption.REPLACE_EXISTING);
                }
            }
            return PersistenceUtils.createOrAppend(typeAnnotation, data, tmpPath, option);
        }
        catch (FileAlreadyExistsException faee) {
            logger.debug("Already have descriptor(s) of type '{}': {}. Skipping.", (Object)new String(typeAnnotation), (Object)outputPath);
        }
        catch (IOException | SecurityException | UnsupportedOperationException e) {
            logger.warn("Could not store descriptor(s) {} of type '{}'", outputPath, new String(typeAnnotation), e);
        }
        catch (Throwable th) {
            logger.warn("Problem storing descriptor(s) {} of type '{}'", outputPath, new String(typeAnnotation), th);
        }
        return false;
    }

    private static boolean createOrAppend(byte[] annotation, byte[] data, Path path, StandardOpenOption option) throws IOException {
        StandardOpenOption appendOption = option;
        Files.createDirectories(path.getParent(), new FileAttribute[0]);
        if (data.length > 0 && data[0] != 64) {
            Files.write(path, annotation, appendOption, StandardOpenOption.CREATE);
            appendOption = StandardOpenOption.APPEND;
        }
        Files.write(path, data, appendOption, StandardOpenOption.CREATE);
        return true;
    }

    public static void cleanDirectory(Path pathToClean) {
        PersistenceUtils.cleanDirectory(pathToClean, -1L);
    }

    public static void cleanDirectory(Path pathToClean, long cutOffMillis) {
        PersistenceUtils.cleanDirectory(pathToClean, cutOffMillis, null);
    }

    public static void cleanDirectory(final Path pathToClean, final long cutOffMillis, final Path pathToExclude) {
        if (!Files.exists(pathToClean, new LinkOption[0])) {
            return;
        }
        logger.info("Cleaning up directory {} with cut-off time {}.", (Object)pathToClean, (Object)Instant.ofEpochMilli(cutOffMillis));
        SimpleFileVisitor<Path> sfv = new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
                if (null == pathToExclude || !pathToExclude.equals(dir)) {
                    return FileVisitResult.CONTINUE;
                }
                return FileVisitResult.SKIP_SUBTREE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                String tempName = file.toString();
                if (cutOffMillis >= 0L && attrs.lastModifiedTime().toMillis() < cutOffMillis) {
                    Files.delete(file);
                } else if (tempName.endsWith(PersistenceUtils.TEMPFIX)) {
                    Path outputPath = Paths.get(tempName.substring(0, tempName.length() - PersistenceUtils.TEMPFIX.length()), new String[0]);
                    Files.deleteIfExists(outputPath);
                    Files.move(file, outputPath, new CopyOption[0]);
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                if (!pathToClean.equals(dir)) {
                    try (Stream<Path> files = Files.list(dir);){
                        if (!files.findFirst().isPresent()) {
                            Files.delete(dir);
                        }
                    }
                }
                return FileVisitResult.CONTINUE;
            }
        };
        try {
            Files.walkFileTree(pathToClean, (FileVisitor<? super Path>)sfv);
        }
        catch (IOException e) {
            logger.warn("Caught I/O exception while cleaning up directory {} with cut-off time {}. Continuing.", pathToClean, Instant.ofEpochMilli(cutOffMillis), e);
        }
    }

    public static String[] dateTimeParts(long dateTime) {
        return PersistenceUtils.dateTimeParts(new Date(dateTime));
    }

    public static String[] dateTimeParts(Date dateTime) {
        SimpleDateFormat printFormat = new SimpleDateFormat("yyyy-MM-dd-HH-mm-ss");
        return printFormat.format(dateTime).split("-");
    }

    public static String dateTime(long dateTime) {
        return PersistenceUtils.dateTime(new Date(dateTime));
    }

    public static String dateTime(Date dateTime) {
        SimpleDateFormat printFormat = new SimpleDateFormat("yyyy-MM-dd-HH-mm-ss");
        return printFormat.format(dateTime);
    }
}

