/*
 * Decompiled with CFR 0.152.
 */
package org.torproject.metrics.collector.indexer;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAmount;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.apache.commons.compress.compressors.bzip2.BZip2CompressorOutputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorOutputStream;
import org.apache.commons.compress.compressors.xz.XZCompressorOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.torproject.metrics.collector.conf.Configuration;
import org.torproject.metrics.collector.conf.ConfigurationException;
import org.torproject.metrics.collector.conf.Key;
import org.torproject.metrics.collector.cron.CollecTorMain;
import org.torproject.metrics.collector.indexer.DirectoryNode;
import org.torproject.metrics.collector.indexer.FileNode;
import org.torproject.metrics.collector.indexer.IndexNode;
import org.torproject.metrics.collector.indexer.IndexerTask;

public class CreateIndexJson
extends CollecTorMain {
    private static final Logger logger = LoggerFactory.getLogger(CreateIndexJson.class);
    private static final TemporalAmount deletionDelay = Duration.ofHours(2L);
    private static final int tarballIndexerThreads = 3;
    private static final int flatFileIndexerThreads = 3;
    private static DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("uuuu-MM-dd HH:mm").withZone(ZoneOffset.UTC);
    private static ObjectMapper objectMapper = new ObjectMapper().setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE).setSerializationInclusion(JsonInclude.Include.NON_EMPTY).setVisibility(PropertyAccessor.ALL, JsonAutoDetect.Visibility.NONE).setVisibility(PropertyAccessor.FIELD, JsonAutoDetect.Visibility.ANY);
    private Path indexedPath;
    private Path htdocsPath;
    private Path indexJsonPath;
    private String basePathString;
    private String buildRevisionString;
    private SortedMap<Path, FileNode> index;
    private ExecutorService tarballsExecutor = Executors.newFixedThreadPool(3);
    private ExecutorService flatFilesExecutor = Executors.newFixedThreadPool(3);

    public CreateIndexJson(Configuration configuration) {
        super(configuration);
    }

    @Override
    public String module() {
        return "updateindex";
    }

    @Override
    protected String syncMarker() {
        return "IndexJson";
    }

    @Override
    public void startProcessing() {
        this.startProcessing(Instant.now());
    }

    protected void startProcessing(Instant now) {
        try {
            this.basePathString = this.config.getProperty(Key.InstanceBaseUrl.name());
            this.indexedPath = this.config.getPath(Key.IndexedPath);
            this.htdocsPath = this.config.getPath(Key.HtdocsPath);
        }
        catch (ConfigurationException e) {
            logger.error("Unable to read one or more configuration values. Not indexing in this execution.", e);
        }
        this.buildRevisionString = this.obtainBuildRevision();
        this.indexJsonPath = this.htdocsPath.resolve(Paths.get("index", "index.json"));
        try {
            this.prepareHtdocsDirectory();
            if (null == this.index) {
                logger.info("Reading index.json file from last execution.");
                this.index = this.readIndex();
            }
            logger.info("Going through indexed/ and adding new files to the index.");
            this.addNewFilesToIndex(this.indexedPath);
            logger.info("Going through htdocs/ and adding links to deleted files to the index.");
            this.addOldLinksToIndex();
            logger.info("Going through the index, scheduling tasks, and updating links.");
            this.scheduleTasksAndUpdateLinks(now);
            logger.info("Writing uncompressed and compressed index.json files to disk.");
            this.writeIndex(this.index, now);
            Runtime rt = Runtime.getRuntime();
            logger.info("Current memory usage is: free = {} B, total = {} B, max = {} B.", rt.freeMemory(), rt.totalMemory(), rt.maxMemory());
            logger.info("Pausing until next index update run.");
        }
        catch (IOException e) {
            logger.error("I/O error while updating index.json files. Trying again in the next execution.", e);
        }
    }

    private void prepareHtdocsDirectory() throws IOException {
        for (Path requiredPath : new Path[]{this.htdocsPath, this.indexJsonPath.getParent()}) {
            if (Files.exists(requiredPath, new LinkOption[0])) continue;
            Files.createDirectories(requiredPath, new FileAttribute[0]);
        }
    }

    private SortedMap<Path, FileNode> readIndex() throws IOException {
        TreeMap<Path, FileNode> index = new TreeMap<Path, FileNode>();
        if (Files.exists(this.indexJsonPath, new LinkOption[0])) {
            IndexNode indexNode = objectMapper.readValue(Files.newInputStream(this.indexJsonPath, new OpenOption[0]), IndexNode.class);
            TreeMap<Path, DirectoryNode> directoryNodes = new TreeMap<Path, DirectoryNode>();
            directoryNodes.put(Paths.get("", new String[0]), indexNode);
            while (!directoryNodes.isEmpty()) {
                Path directoryPath = (Path)directoryNodes.firstKey();
                DirectoryNode directoryNode = (DirectoryNode)directoryNodes.remove(directoryPath);
                if (null != directoryNode.files) {
                    for (FileNode fileNode : directoryNode.files) {
                        Path filePath = this.indexedPath.resolve(directoryPath).resolve(Paths.get(fileNode.path, new String[0]));
                        index.put(filePath, fileNode);
                    }
                }
                if (null == directoryNode.directories) continue;
                boolean isRootDirectory = directoryNode == indexNode;
                for (DirectoryNode subdirectoryNode : directoryNode.directories) {
                    Path subdirectoryPath = isRootDirectory ? Paths.get(subdirectoryNode.path, new String[0]) : directoryPath.resolve(Paths.get(subdirectoryNode.path, new String[0]));
                    directoryNodes.put(subdirectoryPath, subdirectoryNode);
                }
            }
        }
        return index;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    protected String obtainBuildRevision() {
        String buildRevision = null;
        Properties buildProperties = new Properties();
        String propertiesFile = "collector.buildrevision.properties";
        try (InputStream is = this.getClass().getClassLoader().getResourceAsStream(propertiesFile);){
            if (null == is) {
                logger.warn("File {}, which is supposed to contain the build revision string, does not exist in our class path. Writing index.json without the \"build_revision\" field.", (Object)propertiesFile);
                String string = null;
                return string;
            }
            buildProperties.load(is);
            buildRevision = buildProperties.getProperty("collector.build.revision", null);
            return buildRevision;
        }
        catch (IOException e) {
            logger.warn("I/O error while trying to obtain build revision string. Writing index.json without the \"build_revision\" field.");
        }
        return buildRevision;
    }

    private void addNewFilesToIndex(Path path) throws IOException {
        if (!Files.exists(path, new LinkOption[0])) {
            return;
        }
        Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path filePath, BasicFileAttributes basicFileAttributes) {
                if (!filePath.toString().startsWith(".") && !filePath.toString().endsWith(".tmp")) {
                    CreateIndexJson.this.index.putIfAbsent(filePath, new FileNode());
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    private void addOldLinksToIndex() throws IOException {
        final Path htdocsIndexPath = this.indexJsonPath.getParent();
        Files.walkFileTree(this.htdocsPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path linkPath, BasicFileAttributes basicFileAttributes) {
                if (!linkPath.startsWith(htdocsIndexPath)) {
                    Path filePath = CreateIndexJson.this.indexedPath.resolve(CreateIndexJson.this.htdocsPath.relativize(linkPath));
                    CreateIndexJson.this.index.putIfAbsent(filePath, new FileNode());
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    private void scheduleTasksAndUpdateLinks(Instant now) throws IOException {
        int queuedIndexerTasks = 0;
        HashMap<Path, FileNode> indexingResults = new HashMap<Path, FileNode>();
        TreeSet<Path> filesToIndex = new TreeSet<Path>();
        HashMap<Path, Path> linksToCreate = new HashMap<Path, Path>();
        HashSet<FileNode> linksToMarkForDeletion = new HashSet<FileNode>();
        HashMap<Path, Path> linksToDelete = new HashMap<Path, Path>();
        for (Map.Entry<Path, FileNode> e : this.index.entrySet()) {
            Path filePath = e.getKey();
            Path linkPath = this.htdocsPath.resolve(this.indexedPath.relativize(filePath));
            FileNode fileNode = e.getValue();
            if (Files.exists(filePath, new LinkOption[0])) {
                String originalLastModified;
                if (null != fileNode.indexerResult) {
                    if (!fileNode.indexerResult.isDone()) {
                        ++queuedIndexerTasks;
                        continue;
                    }
                    try {
                        fileNode = fileNode.indexerResult.get();
                        indexingResults.put(filePath, fileNode);
                    }
                    catch (InterruptedException | ExecutionException ex) {
                        fileNode.indexerResult = null;
                    }
                }
                if (!(originalLastModified = dateTimeFormatter.format(Files.getLastModifiedTime(filePath, new LinkOption[0]).toInstant())).equals(fileNode.lastModified)) {
                    filesToIndex.add(filePath);
                    continue;
                }
                if (!Files.exists(linkPath, new LinkOption[0])) {
                    linksToCreate.put(linkPath, filePath);
                    if (null == fileNode.markedForDeletion) continue;
                    fileNode.markedForDeletion = null;
                    continue;
                }
                String linkLastModified = dateTimeFormatter.format(Files.getLastModifiedTime(linkPath, new LinkOption[0]).toInstant());
                if (linkLastModified.equals(fileNode.lastModified)) continue;
                linksToCreate.put(linkPath, filePath);
                continue;
            }
            if (null == fileNode.markedForDeletion) {
                linksToMarkForDeletion.add(fileNode);
                continue;
            }
            if (!fileNode.markedForDeletion.isBefore(now.minus(deletionDelay))) continue;
            linksToDelete.put(linkPath, filePath);
        }
        if (queuedIndexerTasks > 0) {
            logger.info("Counting {} file(s) being currently indexed or in the queue.", (Object)queuedIndexerTasks);
        }
        this.updateIndex(indexingResults);
        this.scheduleTasks(filesToIndex);
        this.createLinks(linksToCreate);
        this.markForDeletion(linksToMarkForDeletion, now);
        this.deleteLinks(linksToDelete);
    }

    private void updateIndex(Map<Path, FileNode> indexResults) {
        if (!indexResults.isEmpty()) {
            logger.info("Updating {} index entries with index results.", (Object)indexResults.size());
            this.index.putAll(indexResults);
        }
    }

    private void scheduleTasks(SortedSet<Path> filesToIndex) {
        if (!filesToIndex.isEmpty()) {
            logger.info("Scheduling {} indexer task(s).", (Object)filesToIndex.size());
            for (Path fileToIndex : filesToIndex) {
                IndexerTask indexerTask = this.createIndexerTask(fileToIndex);
                if (fileToIndex.getFileName().toString().endsWith(".tar.xz")) {
                    ((FileNode)this.index.get((Object)fileToIndex)).indexerResult = this.tarballsExecutor.submit(indexerTask);
                    continue;
                }
                ((FileNode)this.index.get((Object)fileToIndex)).indexerResult = this.flatFilesExecutor.submit(indexerTask);
            }
        }
    }

    protected IndexerTask createIndexerTask(Path fileToIndex) {
        return new IndexerTask(fileToIndex);
    }

    private void createLinks(Map<Path, Path> linksToCreate) throws IOException {
        if (!linksToCreate.isEmpty()) {
            logger.info("Creating {} new link(s).", (Object)linksToCreate.size());
            for (Map.Entry<Path, Path> e : linksToCreate.entrySet()) {
                Path linkPath = e.getKey();
                Path originalPath = e.getValue();
                Files.createDirectories(linkPath.getParent(), new FileAttribute[0]);
                Files.deleteIfExists(linkPath);
                Files.createLink(linkPath, originalPath);
            }
        }
    }

    private void markForDeletion(Set<FileNode> linksToMarkForDeletion, Instant now) {
        if (!linksToMarkForDeletion.isEmpty()) {
            logger.info("Marking {} old link(s) for deletion.", (Object)linksToMarkForDeletion.size());
            for (FileNode fileNode : linksToMarkForDeletion) {
                fileNode.markedForDeletion = now;
            }
        }
    }

    private void deleteLinks(Map<Path, Path> linksToDelete) throws IOException {
        if (!linksToDelete.isEmpty()) {
            logger.info("Deleting {} old link(s).", (Object)linksToDelete.size());
            for (Map.Entry<Path, Path> e : linksToDelete.entrySet()) {
                Path linkPath = e.getKey();
                Path originalPath = e.getValue();
                Files.deleteIfExists(linkPath);
                this.index.remove(originalPath);
            }
        }
    }

    private void writeIndex(SortedMap<Path, FileNode> index, Instant now) throws IOException {
        IndexNode indexNode = new IndexNode();
        indexNode.indexCreated = dateTimeFormatter.format(now);
        indexNode.buildRevision = this.buildRevisionString;
        indexNode.path = this.basePathString;
        TreeMap<Path, DirectoryNode> directoryNodes = new TreeMap<Path, DirectoryNode>();
        for (Map.Entry<Path, FileNode> indexEntry : index.entrySet()) {
            Path filePath = this.indexedPath.relativize(indexEntry.getKey());
            FileNode fileNode = indexEntry.getValue();
            if (null == fileNode.lastModified || null != fileNode.markedForDeletion) continue;
            Path directoryPath = null;
            DirectoryNode parentDirectoryNode = indexNode;
            if (null != filePath.getParent()) {
                for (Path pathPart : filePath.getParent()) {
                    DirectoryNode directoryNode = (DirectoryNode)directoryNodes.get(directoryPath = null == directoryPath ? pathPart : directoryPath.resolve(pathPart));
                    if (null == directoryNode) {
                        directoryNode = new DirectoryNode();
                        directoryNode.path = pathPart.toString();
                        if (null == parentDirectoryNode.directories) {
                            parentDirectoryNode.directories = new ArrayList<DirectoryNode>();
                        }
                        parentDirectoryNode.directories.add(directoryNode);
                        directoryNodes.put(directoryPath, directoryNode);
                    }
                    parentDirectoryNode = directoryNode;
                }
            }
            if (null == parentDirectoryNode.files) {
                parentDirectoryNode.files = new ArrayList<FileNode>();
            }
            parentDirectoryNode.files.add(fileNode);
        }
        Path htdocsIndexPath = this.indexJsonPath.getParent();
        try (OutputStream uncompressed = Files.newOutputStream(htdocsIndexPath.resolve(".index.json.tmp"), new OpenOption[0]);
             BZip2CompressorOutputStream bz2Compressed = new BZip2CompressorOutputStream(Files.newOutputStream(htdocsIndexPath.resolve("index.json.bz2"), new OpenOption[0]));
             GzipCompressorOutputStream gzCompressed = new GzipCompressorOutputStream(Files.newOutputStream(htdocsIndexPath.resolve("index.json.gz"), new OpenOption[0]));
             XZCompressorOutputStream xzCompressed = new XZCompressorOutputStream(Files.newOutputStream(htdocsIndexPath.resolve("index.json.xz"), new OpenOption[0]));){
            objectMapper.writeValue(uncompressed, (Object)indexNode);
            objectMapper.writeValue(bz2Compressed, (Object)indexNode);
            objectMapper.writeValue(gzCompressed, (Object)indexNode);
            objectMapper.writeValue(xzCompressed, (Object)indexNode);
        }
        Files.move(htdocsIndexPath.resolve(".index.json.tmp"), this.indexJsonPath, StandardCopyOption.REPLACE_EXISTING);
    }
}

