/*
 * Decompiled with CFR 0.152.
 */
package org.torproject.metrics.collector.bridgedescs;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Paths;
import java.security.GeneralSecurityException;
import java.security.SecureRandom;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.SortedMap;
import java.util.Stack;
import java.util.TimeZone;
import java.util.TreeMap;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.digest.DigestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.torproject.descriptor.BridgeExtraInfoDescriptor;
import org.torproject.descriptor.BridgeNetworkStatus;
import org.torproject.descriptor.BridgeServerDescriptor;
import org.torproject.metrics.collector.bridgedescs.BridgeDescriptorParser;
import org.torproject.metrics.collector.bridgedescs.BridgeSnapshotReader;
import org.torproject.metrics.collector.bridgedescs.DescriptorBuilder;
import org.torproject.metrics.collector.conf.Annotation;
import org.torproject.metrics.collector.conf.Configuration;
import org.torproject.metrics.collector.conf.ConfigurationException;
import org.torproject.metrics.collector.conf.Key;
import org.torproject.metrics.collector.cron.CollecTorMain;

public class SanitizedBridgesWriter
extends CollecTorMain {
    private static final Logger logger = LoggerFactory.getLogger(SanitizedBridgesWriter.class);
    private static final String BRIDGE_DESCRIPTORS = "bridge-descriptors";
    private String rsyncCatString;
    private File bridgeDirectoriesDirectory;
    private File sanitizedBridgesDirectory;
    private boolean replaceIpAddressesWithHashes;
    private boolean persistenceProblemWithSecrets;
    private SortedMap<String, byte[]> secretsForHashingIpAddresses;
    private String bridgeSanitizingCutOffTimestamp;
    private boolean haveWarnedAboutInterval;
    private File bridgeIpSecretsFile;
    private SecureRandom secureRandom;
    private String outputPathName;
    private String recentPathName;
    private String maxNetworkStatusPublishedTime = "1970-01-01 00:00:00";
    private String maxServerDescriptorPublishedTime = "1970-01-01 00:00:00";
    private String maxExtraInfoDescriptorPublishedTime = "1970-01-01 00:00:00";

    public SanitizedBridgesWriter(Configuration config) {
        super(config);
        this.mapPathDescriptors.put("recent/bridge-descriptors/statuses", BridgeNetworkStatus.class);
        this.mapPathDescriptors.put("recent/bridge-descriptors/server-descriptors", BridgeServerDescriptor.class);
        this.mapPathDescriptors.put("recent/bridge-descriptors/extra-infos", BridgeExtraInfoDescriptor.class);
    }

    @Override
    public String module() {
        return "bridgedescs";
    }

    @Override
    protected String syncMarker() {
        return "Bridge";
    }

    @Override
    protected void startProcessing() throws ConfigurationException {
        LocalDateTime configuredBridgeSanitizingCutOffDateTime;
        this.outputPathName = Paths.get(this.config.getPath(Key.OutputPath).toString(), BRIDGE_DESCRIPTORS).toString();
        this.recentPathName = Paths.get(this.config.getPath(Key.RecentPath).toString(), BRIDGE_DESCRIPTORS).toString();
        File bridgeDirectoriesDirectory = this.config.getPath(Key.BridgeLocalOrigins).toFile();
        File sanitizedBridgesDirectory = new File(this.outputPathName);
        File statsDirectory = this.config.getPath(Key.StatsPath).toFile();
        if (bridgeDirectoriesDirectory == null || sanitizedBridgesDirectory == null || statsDirectory == null) {
            throw new ConfigurationException("BridgeSnapshotsDirectory, SanitizedBridgesWriteDirectory, StatsPath should be set. Please, edit the 'collector.properties' file.");
        }
        this.bridgeDirectoriesDirectory = bridgeDirectoriesDirectory;
        this.sanitizedBridgesDirectory = sanitizedBridgesDirectory;
        this.replaceIpAddressesWithHashes = this.config.getBool(Key.ReplaceIpAddressesWithHashes);
        SimpleDateFormat rsyncCatFormat = new SimpleDateFormat("yyyy-MM-dd-HH-mm-ss");
        rsyncCatFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        this.rsyncCatString = rsyncCatFormat.format(System.currentTimeMillis());
        if (this.replaceIpAddressesWithHashes) {
            try {
                this.secureRandom = SecureRandom.getInstance("SHA1PRNG", "SUN");
            }
            catch (GeneralSecurityException e) {
                logger.warn("Could not initialize secure random number generator! Not calculating any IP address hashes in this execution!", e);
                this.persistenceProblemWithSecrets = true;
            }
        }
        this.secretsForHashingIpAddresses = new TreeMap<String, byte[]>();
        this.bridgeIpSecretsFile = new File(statsDirectory, "bridge-ip-secrets");
        if (this.bridgeIpSecretsFile.exists()) {
            try (BufferedReader br = new BufferedReader(new FileReader(this.bridgeIpSecretsFile));){
                String line;
                while ((line = br.readLine()) != null) {
                    String[] parts = line.split(",");
                    if (line.length() != "yyyy-MM,".length() + 62 && line.length() != "yyyy-MM,".length() + 100 && line.length() != "yyyy-MM,".length() + 166 || parts.length != 2) {
                        logger.warn("Invalid line in bridge-ip-secrets file starting with '{}'! Not calculating any IP address hashes in this execution!", (Object)line.substring(0, 7));
                        this.persistenceProblemWithSecrets = true;
                        break;
                    }
                    String month = parts[0];
                    byte[] secret = Hex.decodeHex(parts[1].toCharArray());
                    this.secretsForHashingIpAddresses.put(month, secret);
                }
                if (!this.persistenceProblemWithSecrets) {
                    logger.debug("Read {} secrets for hashing bridge IP addresses.", (Object)this.secretsForHashingIpAddresses.size());
                }
            }
            catch (DecoderException e) {
                logger.warn("Failed to decode hex string in {}! Not calculating any IP address hashes in this execution!", (Object)this.bridgeIpSecretsFile, (Object)e);
                this.persistenceProblemWithSecrets = true;
            }
            catch (IOException e) {
                logger.warn("Failed to read {}! Not calculating any IP address hashes in this execution!", (Object)this.bridgeIpSecretsFile, (Object)e);
                this.persistenceProblemWithSecrets = true;
            }
        }
        long limitBridgeSanitizingIntervalDays = this.config.getInt(Key.BridgeDescriptorMappingsLimit);
        LocalDateTime bridgeSanitizingCutOffDateTime = LocalDateTime.of(1999, 12, 31, 23, 59, 59);
        if (limitBridgeSanitizingIntervalDays >= 0L && (configuredBridgeSanitizingCutOffDateTime = LocalDateTime.now(ZoneOffset.UTC).minusDays(limitBridgeSanitizingIntervalDays)).isAfter(bridgeSanitizingCutOffDateTime)) {
            bridgeSanitizingCutOffDateTime = configuredBridgeSanitizingCutOffDateTime;
        }
        this.bridgeSanitizingCutOffTimestamp = bridgeSanitizingCutOffDateTime.format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
        logger.info("Using cut-off datetime '{}' for secrets.", (Object)this.bridgeSanitizingCutOffTimestamp);
        BridgeDescriptorParser bdp = new BridgeDescriptorParser(this);
        new BridgeSnapshotReader(bdp, this.bridgeDirectoriesDirectory, statsDirectory);
        this.finishWriting();
        this.checkStaleDescriptors();
        this.cleanUpRsyncDirectory();
    }

    private String scrubOrAddress(String orAddress, byte[] fingerprintBytes, String published) throws IOException {
        if (!orAddress.contains(":")) {
            return null;
        }
        String addressPart = orAddress.substring(0, orAddress.lastIndexOf(":"));
        String portPart = orAddress.substring(orAddress.lastIndexOf(":") + 1);
        String scrubbedAddressPart = addressPart.startsWith("[") ? this.scrubIpv6Address(addressPart, fingerprintBytes, published) : this.scrubIpv4Address(addressPart, fingerprintBytes, published);
        String scrubbedPort = this.scrubTcpPort(portPart, fingerprintBytes, published);
        return scrubbedAddressPart == null ? null : scrubbedAddressPart + ":" + scrubbedPort;
    }

    private String scrubIpv4Address(String address, byte[] fingerprintBytes, String published) throws IOException {
        if (this.replaceIpAddressesWithHashes) {
            if (this.persistenceProblemWithSecrets) {
                return null;
            }
            byte[] hashInput = new byte[55];
            String[] ipParts = address.split("\\.");
            for (int i = 0; i < 4; ++i) {
                hashInput[i] = (byte)Integer.parseInt(ipParts[i]);
            }
            System.arraycopy(fingerprintBytes, 0, hashInput, 4, 20);
            String month = published.substring(0, "yyyy-MM".length());
            byte[] secret = this.getSecretForMonth(month);
            System.arraycopy(secret, 0, hashInput, 24, 31);
            byte[] hashOutput = DigestUtils.sha256(hashInput);
            return "10." + (hashOutput[0] + 256) % 256 + "." + (hashOutput[1] + 256) % 256 + "." + (hashOutput[2] + 256) % 256;
        }
        return "127.0.0.1";
    }

    private String scrubIpv6Address(String address, byte[] fingerprintBytes, String published) throws IOException {
        StringBuilder sb = new StringBuilder("[fd9f:2e19:3bcf::");
        if (this.replaceIpAddressesWithHashes) {
            byte[] ipBytes;
            if (this.persistenceProblemWithSecrets) {
                return null;
            }
            String[] doubleColonSeparatedParts = address.substring(1, address.length() - 1).split("::", -1);
            if (doubleColonSeparatedParts.length > 2) {
                return null;
            }
            ArrayList<String> hexParts = new ArrayList<String>();
            for (String doubleColonSeparatedPart : doubleColonSeparatedParts) {
                StringBuilder hexPart = new StringBuilder();
                String[] parts = doubleColonSeparatedPart.split(":", -1);
                if (parts.length < 1 || parts.length > 8) {
                    return null;
                }
                for (String part : parts) {
                    if (part.contains(".")) {
                        String[] ipParts = part.split("\\.");
                        byte[] ipv4Bytes = new byte[4];
                        if (ipParts.length != 4) {
                            return null;
                        }
                        for (int m = 0; m < 4; ++m) {
                            ipv4Bytes[m] = (byte)Integer.parseInt(ipParts[m]);
                        }
                        hexPart.append(Hex.encodeHexString(ipv4Bytes));
                        continue;
                    }
                    if (part.length() > 4) {
                        return null;
                    }
                    for (int k = part.length(); k < 4; ++k) {
                        hexPart.append("0");
                    }
                    hexPart.append(part);
                }
                hexParts.add(hexPart.toString());
            }
            StringBuilder hex = new StringBuilder();
            hex.append((String)hexParts.get(0));
            if (hexParts.size() == 2) {
                for (int i = 32 - ((String)hexParts.get(0)).length() - ((String)hexParts.get(1)).length(); i > 0; --i) {
                    hex.append("0");
                }
                hex.append((String)hexParts.get(1));
            }
            try {
                ipBytes = Hex.decodeHex(hex.toString().toCharArray());
            }
            catch (DecoderException e) {
                return null;
            }
            if (ipBytes.length != 16) {
                return null;
            }
            byte[] hashInput = new byte[55];
            System.arraycopy(ipBytes, 0, hashInput, 0, 16);
            System.arraycopy(fingerprintBytes, 0, hashInput, 16, 20);
            String month = published.substring(0, "yyyy-MM".length());
            byte[] secret = this.getSecretForMonth(month);
            System.arraycopy(secret, 31, hashInput, 36, 19);
            String hashOutput = DigestUtils.sha256Hex(hashInput);
            sb.append(hashOutput, hashOutput.length() - 6, hashOutput.length() - 4);
            sb.append(":");
            sb.append(hashOutput.substring(hashOutput.length() - 4));
        }
        sb.append("]");
        return sb.toString();
    }

    private String scrubTcpPort(String portString, byte[] fingerprintBytes, String published) throws IOException {
        if (portString.equals("0")) {
            return "0";
        }
        if (this.replaceIpAddressesWithHashes) {
            if (this.persistenceProblemWithSecrets) {
                return null;
            }
            byte[] hashInput = new byte[55];
            int portNumber = Integer.parseInt(portString);
            hashInput[0] = (byte)(portNumber >> 8);
            hashInput[1] = (byte)portNumber;
            System.arraycopy(fingerprintBytes, 0, hashInput, 2, 20);
            String month = published.substring(0, "yyyy-MM".length());
            byte[] secret = this.getSecretForMonth(month);
            System.arraycopy(secret, 50, hashInput, 22, 33);
            byte[] hashOutput = DigestUtils.sha256(hashInput);
            int hashedPort = ((hashOutput[0] & 0xFF) << 8 | hashOutput[1] & 0xFF) >> 2 | 0xC000;
            return String.valueOf(hashedPort);
        }
        return "1";
    }

    private byte[] getSecretForMonth(String month) throws IOException {
        if (!this.secretsForHashingIpAddresses.containsKey(month) || ((byte[])this.secretsForHashingIpAddresses.get(month)).length < 83) {
            byte[] secret = new byte[83];
            this.secureRandom.nextBytes(secret);
            if (this.secretsForHashingIpAddresses.containsKey(month)) {
                System.arraycopy(this.secretsForHashingIpAddresses.get(month), 0, secret, 0, ((byte[])this.secretsForHashingIpAddresses.get(month)).length);
            }
            if (month.compareTo(this.bridgeSanitizingCutOffTimestamp) < 0) {
                logger.warn("Generated a secret that we won't make persistent, because it's outside our bridge descriptor sanitizing interval.");
            } else {
                try {
                    if (!this.bridgeIpSecretsFile.exists()) {
                        this.bridgeIpSecretsFile.getParentFile().mkdirs();
                    }
                    BufferedWriter bw = new BufferedWriter(new FileWriter(this.bridgeIpSecretsFile, this.bridgeIpSecretsFile.exists()));
                    bw.write(month + "," + Hex.encodeHexString(secret) + "\n");
                    bw.close();
                }
                catch (IOException e) {
                    logger.warn("Could not store new secret to disk! Not calculating any IP address or TCP port hashes in this execution!", e);
                    this.persistenceProblemWithSecrets = true;
                    throw new IOException(e);
                }
            }
            this.secretsForHashingIpAddresses.put(month, secret);
        }
        return (byte[])this.secretsForHashingIpAddresses.get(month);
    }

    public void sanitizeAndStoreNetworkStatus(byte[] data, String publicationTime, String authorityFingerprint) {
        if (this.persistenceProblemWithSecrets) {
            return;
        }
        if (publicationTime.compareTo(this.maxNetworkStatusPublishedTime) > 0) {
            this.maxNetworkStatusPublishedTime = publicationTime;
        }
        if (this.bridgeSanitizingCutOffTimestamp.compareTo(publicationTime) > 0) {
            String text = "Sanitizing and storing network status with publication time outside our descriptor sanitizing interval.";
            if (this.haveWarnedAboutInterval) {
                logger.debug(text);
            } else {
                logger.warn(text);
                this.haveWarnedAboutInterval = true;
            }
        }
        DescriptorBuilder header = new DescriptorBuilder();
        boolean includesFingerprintLine = false;
        TreeMap<String, String> scrubbedLines = new TreeMap<String, String>();
        try {
            String scrubbedLine;
            String line;
            DescriptorBuilder scrubbed = new DescriptorBuilder();
            BufferedReader br = new BufferedReader(new StringReader(new String(data, "US-ASCII")));
            String mostRecentDescPublished = null;
            byte[] fingerprintBytes = null;
            String descPublicationTime = null;
            String hashedBridgeIdentityHex = null;
            while ((line = br.readLine()) != null) {
                if (line.startsWith("published ")) {
                    publicationTime = line.substring("published ".length());
                    continue;
                }
                if (line.startsWith("flag-thresholds ")) {
                    header.append(line).newLine();
                    continue;
                }
                if (line.startsWith("fingerprint ")) {
                    if (!("fingerprint " + authorityFingerprint).equals(line)) {
                        logger.warn("Mismatch between authority fingerprint expected from file name ({}) and parsed from \"fingerprint\" line (\"{}\").", (Object)authorityFingerprint, (Object)line);
                        return;
                    }
                    header.append(line).newLine();
                    includesFingerprintLine = true;
                    continue;
                }
                if (line.startsWith("r ")) {
                    String[] parts;
                    if (scrubbed.hasContent()) {
                        scrubbedLine = scrubbed.toString();
                        scrubbedLines.put(hashedBridgeIdentityHex, scrubbedLine);
                        scrubbed = new DescriptorBuilder();
                    }
                    if ((parts = line.split(" ")).length < 9) {
                        logger.warn("Illegal line '{}' in bridge network status.  Skipping descriptor.", (Object)line);
                        return;
                    }
                    if (!Base64.isBase64(parts[2])) {
                        logger.warn("Illegal base64 character in r line '{}'.  Skipping descriptor.", (Object)parts[2]);
                        return;
                    }
                    fingerprintBytes = Base64.decodeBase64(parts[2] + "==");
                    descPublicationTime = parts[4] + " " + parts[5];
                    String address = parts[6];
                    String orPort = parts[7];
                    String dirPort = parts[8];
                    if (descPublicationTime.compareTo(publicationTime) <= 0 && (mostRecentDescPublished == null || descPublicationTime.compareTo(mostRecentDescPublished) > 0)) {
                        mostRecentDescPublished = descPublicationTime;
                    }
                    byte[] hashedBridgeIdentity = DigestUtils.sha1(fingerprintBytes);
                    String hashedBridgeIdentityBase64 = Base64.encodeBase64String(hashedBridgeIdentity).substring(0, 27);
                    hashedBridgeIdentityHex = Hex.encodeHexString(hashedBridgeIdentity);
                    String descriptorIdentifier = parts[3];
                    String hashedDescriptorIdentifier = Base64.encodeBase64String(DigestUtils.sha1(Base64.decodeBase64(descriptorIdentifier + "=="))).substring(0, 27);
                    String scrubbedAddress = this.scrubIpv4Address(address, fingerprintBytes, descPublicationTime);
                    String nickname = parts[1];
                    String scrubbedOrPort = this.scrubTcpPort(orPort, fingerprintBytes, descPublicationTime);
                    String scrubbedDirPort = this.scrubTcpPort(dirPort, fingerprintBytes, descPublicationTime);
                    scrubbed.append("r ").append(nickname).space().append(hashedBridgeIdentityBase64).space().append(hashedDescriptorIdentifier).space().append(descPublicationTime).space().append(scrubbedAddress).space().append(scrubbedOrPort).space().append(scrubbedDirPort).newLine();
                    continue;
                }
                if (line.startsWith("a ")) {
                    String scrubbedOrAddress = this.scrubOrAddress(line.substring("a ".length()), fingerprintBytes, descPublicationTime);
                    if (scrubbedOrAddress != null) {
                        scrubbed.append("a ").append(scrubbedOrAddress).newLine();
                        continue;
                    }
                    logger.warn("Invalid address in line '{}' in bridge network status.  Skipping line!", (Object)line);
                    continue;
                }
                if (line.startsWith("s ") || line.equals("s") || line.startsWith("w ") || line.equals("w") || line.startsWith("p ") || line.equals("p")) {
                    scrubbed.append(line).newLine();
                    continue;
                }
                logger.debug("Unknown line '{}' in bridge network status. Not writing to disk!", (Object)line);
                return;
            }
            br.close();
            if (scrubbed.hasContent()) {
                scrubbedLine = scrubbed.toString();
                scrubbedLines.put(hashedBridgeIdentityHex, scrubbedLine);
            }
            if (!includesFingerprintLine) {
                header.append("fingerprint ").append(authorityFingerprint).newLine();
            }
            SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            formatter.setTimeZone(TimeZone.getTimeZone("UTC"));
            if (null == mostRecentDescPublished) {
                logger.warn("The bridge network status published at {} does not contain a single entry. Please ask the bridge authority operator to check!", (Object)publicationTime);
            } else if (formatter.parse(publicationTime).getTime() - formatter.parse(mostRecentDescPublished).getTime() > 3600000L) {
                logger.warn("The most recent descriptor in the bridge network status published at {} was published at {} which is more than 1 hour before the status. This is a sign for the status being stale. Please check!", (Object)publicationTime, (Object)mostRecentDescPublished);
            }
        }
        catch (ParseException e) {
            logger.warn("Could not parse timestamp in bridge network status.", e);
            return;
        }
        catch (IOException e) {
            logger.warn("Could not parse bridge network status.", e);
            return;
        }
        try {
            File[] outputFiles;
            String syear = publicationTime.substring(0, 4);
            String smonth = publicationTime.substring(5, 7);
            String sday = publicationTime.substring(8, 10);
            String stime = publicationTime.substring(11, 13) + publicationTime.substring(14, 16) + publicationTime.substring(17, 19);
            File tarballFile = new File(this.sanitizedBridgesDirectory.getAbsolutePath() + "/" + syear + "/" + smonth + "/statuses/" + sday + "/" + syear + smonth + sday + "-" + stime + "-" + authorityFingerprint);
            File rsyncFile = new File(this.recentPathName, "statuses/" + tarballFile.getName());
            for (File outputFile : outputFiles = new File[]{tarballFile, rsyncFile}) {
                outputFile.getParentFile().mkdirs();
                BufferedWriter bw = new BufferedWriter(new FileWriter(outputFile));
                bw.write(Annotation.Status.toString());
                bw.write("published " + publicationTime + "\n");
                bw.write(header.toString());
                for (String scrubbed : scrubbedLines.values()) {
                    bw.write(scrubbed);
                }
                bw.close();
            }
        }
        catch (IOException e) {
            logger.warn("Could not write sanitized bridge network status to disk.", e);
        }
    }

    public void sanitizeAndStoreServerDescriptor(byte[] data) {
        if (this.persistenceProblemWithSecrets) {
            return;
        }
        String address = null;
        String published = null;
        byte[] fingerprintBytes = null;
        StringBuilder scrubbedAddress = null;
        HashMap<StringBuilder, String> scrubbedTcpPorts = new HashMap<StringBuilder, String>();
        HashMap<StringBuilder, String> scrubbedIpAddressesAndTcpPorts = new HashMap<StringBuilder, String>();
        String masterKeyEd25519FromIdentityEd25519 = null;
        DescriptorBuilder scrubbed = new DescriptorBuilder();
        try (BufferedReader br = new BufferedReader(new StringReader(new String(data, "US-ASCII")));){
            Object line;
            scrubbed.append(Annotation.BridgeServer.toString());
            String masterKeyEd25519 = null;
            boolean skipCrypto = false;
            while ((line = br.readLine()) != null) {
                String[] parts;
                if (skipCrypto && !((String)line).startsWith("-----END ")) continue;
                if (((String)line).startsWith("router ")) {
                    parts = ((String)line).split(" ");
                    if (parts.length != 6) {
                        logger.warn("Invalid router line: '{}'.  Skipping.", line);
                        return;
                    }
                    address = parts[2];
                    scrubbedAddress = new StringBuilder();
                    StringBuilder scrubbedOrPort = new StringBuilder();
                    scrubbedTcpPorts.put(scrubbedOrPort, parts[3]);
                    StringBuilder scrubbedDirPort = new StringBuilder();
                    scrubbedTcpPorts.put(scrubbedDirPort, parts[4]);
                    StringBuilder scrubbedSocksPort = new StringBuilder();
                    scrubbedTcpPorts.put(scrubbedSocksPort, parts[5]);
                    scrubbed.append("router ").append(parts[1]).space().append(scrubbedAddress).space().append(scrubbedOrPort).space().append(scrubbedDirPort).space().append(scrubbedSocksPort).newLine();
                    continue;
                }
                if (((String)line).startsWith("or-address ")) {
                    String orAddress = ((String)line).substring("or-address ".length());
                    StringBuilder scrubbedOrAddress = new StringBuilder();
                    scrubbedIpAddressesAndTcpPorts.put(scrubbedOrAddress, orAddress);
                    scrubbed.append("or-address ").append(scrubbedOrAddress).newLine();
                    continue;
                }
                if (((String)line).startsWith("published ")) {
                    published = ((String)line).substring("published ".length());
                    if (published.compareTo(this.maxServerDescriptorPublishedTime) > 0) {
                        this.maxServerDescriptorPublishedTime = published;
                    }
                    if (this.bridgeSanitizingCutOffTimestamp.compareTo(published) > 0) {
                        String text = "Sanitizing and storing server descriptor with publication time outside our descriptor sanitizing interval.";
                        if (this.haveWarnedAboutInterval) {
                            logger.debug(text);
                        } else {
                            logger.warn(text);
                            this.haveWarnedAboutInterval = true;
                        }
                    }
                    scrubbed.append((String)line).newLine();
                    continue;
                }
                if (((String)line).startsWith("opt fingerprint ") || ((String)line).startsWith("fingerprint ")) {
                    String fingerprint = ((String)line).substring(((String)line).startsWith("opt ") ? "opt fingerprint".length() : "fingerprint".length()).replaceAll(" ", "").toLowerCase();
                    fingerprintBytes = Hex.decodeHex(fingerprint.toCharArray());
                    String hashedBridgeIdentity = DigestUtils.sha1Hex(fingerprintBytes).toLowerCase();
                    scrubbed.append(((String)line).startsWith("opt ") ? "opt " : "").append("fingerprint");
                    for (int i = 0; i < hashedBridgeIdentity.length() / 4; ++i) {
                        scrubbed.space().append(hashedBridgeIdentity.substring(4 * i, 4 * (i + 1)).toUpperCase());
                    }
                    scrubbed.newLine();
                    continue;
                }
                if (((String)line).startsWith("contact ")) {
                    scrubbed.append("contact somebody").newLine();
                    continue;
                }
                if (((String)line).startsWith("router-signature")) {
                    break;
                }
                if (((String)line).startsWith("opt extra-info-digest ") || ((String)line).startsWith("extra-info-digest ")) {
                    parts = ((String)line).split(" ");
                    if (((String)line).startsWith("opt ")) {
                        scrubbed.append("opt ");
                        parts = ((String)line).substring(4).split(" ");
                    }
                    if (parts.length > 3) {
                        logger.warn("extra-info-digest line contains more arguments thanexpected: '{}'.  Skipping descriptor.", line);
                        return;
                    }
                    scrubbed.append("extra-info-digest ").append(DigestUtils.sha1Hex(Hex.decodeHex(parts[1].toCharArray())).toUpperCase());
                    if (parts.length > 2) {
                        if (!Base64.isBase64(parts[2])) {
                            logger.warn("Illegal base64 character in extra-info-digest line '{}'.  Skipping descriptor.", line);
                            return;
                        }
                        scrubbed.space().append(Base64.encodeBase64String(DigestUtils.sha256(Base64.decodeBase64(parts[2]))).replaceAll("=", ""));
                    }
                    scrubbed.newLine();
                    continue;
                }
                if (((String)line).startsWith("reject ")) {
                    if (address != null && ((String)line).startsWith("reject " + address)) {
                        scrubbed.append("reject ").append(scrubbedAddress).append(((String)line).substring("reject ".length() + address.length())).newLine();
                        continue;
                    }
                    scrubbed.append((String)line).newLine();
                    continue;
                }
                if (((String)line).equals("identity-ed25519")) {
                    StringBuilder sb = new StringBuilder();
                    while ((line = br.readLine()) != null && !((String)line).equals("-----END ED25519 CERT-----")) {
                        if (((String)line).equals("-----BEGIN ED25519 CERT-----")) continue;
                        sb.append((String)line);
                    }
                    masterKeyEd25519FromIdentityEd25519 = this.parseMasterKeyEd25519FromIdentityEd25519(sb.toString());
                    if (masterKeyEd25519FromIdentityEd25519 == null) {
                        logger.warn("Could not parse master-key-ed25519 from identity-ed25519.  Skipping descriptor.");
                        return;
                    }
                    String[] sha256MasterKeyEd25519 = Base64.encodeBase64String(DigestUtils.sha256(Base64.decodeBase64(masterKeyEd25519FromIdentityEd25519 + "="))).replaceAll("=", "");
                    scrubbed.append("master-key-ed25519 ").append((String)sha256MasterKeyEd25519).newLine();
                    if (masterKeyEd25519 == null || masterKeyEd25519.equals(masterKeyEd25519FromIdentityEd25519)) continue;
                    logger.warn("Mismatch between identity-ed25519 and master-key-ed25519.  Skipping.");
                    return;
                }
                if (((String)line).startsWith("master-key-ed25519 ")) {
                    masterKeyEd25519 = ((String)line).substring(((String)line).indexOf(" ") + 1);
                    if (masterKeyEd25519FromIdentityEd25519 == null || masterKeyEd25519FromIdentityEd25519.equals(masterKeyEd25519)) continue;
                    logger.warn("Mismatch between identity-ed25519 and master-key-ed25519.  Skipping.");
                    return;
                }
                if (((String)line).startsWith("accept ") || ((String)line).startsWith("platform ") || ((String)line).startsWith("opt protocols ") || ((String)line).startsWith("protocols ") || ((String)line).startsWith("proto ") || ((String)line).startsWith("uptime ") || ((String)line).startsWith("bandwidth ") || ((String)line).startsWith("opt hibernating ") || ((String)line).startsWith("hibernating ") || ((String)line).startsWith("ntor-onion-key ") || ((String)line).equals("opt hidden-service-dir") || ((String)line).equals("hidden-service-dir") || ((String)line).equals("opt caches-extra-info") || ((String)line).equals("caches-extra-info") || ((String)line).equals("opt allow-single-hop-exits") || ((String)line).equals("allow-single-hop-exits") || ((String)line).startsWith("ipv6-policy ") || ((String)line).equals("tunnelled-dir-server") || ((String)line).startsWith("bridge-distribution-request ")) {
                    scrubbed.append((String)line).newLine();
                    continue;
                }
                if (((String)line).startsWith("family ")) {
                    DescriptorBuilder familyLine = new DescriptorBuilder("family");
                    for (String s : ((String)line).substring(7).split(" ")) {
                        if (s.startsWith("$")) {
                            familyLine.append(" $").append(DigestUtils.sha1Hex(Hex.decodeHex(s.substring(1).toCharArray())).toUpperCase());
                            continue;
                        }
                        familyLine.space().append(s);
                    }
                    scrubbed.append(familyLine.toString()).newLine();
                    continue;
                }
                if (((String)line).startsWith("@purpose ")) continue;
                if (((String)line).startsWith("-----BEGIN ") || ((String)line).equals("onion-key") || ((String)line).equals("signing-key") || ((String)line).equals("onion-key-crosscert") || ((String)line).startsWith("ntor-onion-key-crosscert ")) {
                    skipCrypto = true;
                    continue;
                }
                if (((String)line).startsWith("-----END ")) {
                    skipCrypto = false;
                    continue;
                }
                if (((String)line).startsWith("router-sig-ed25519 ")) continue;
                logger.warn("Unrecognized line '{}'. Skipping.", line);
                return;
            }
        }
        catch (Exception e) {
            logger.warn("Could not parse server descriptor.", e);
            return;
        }
        if (null == address || null == fingerprintBytes || null == published) {
            logger.warn("Missing either of the following lines that are required to sanitize this server bridge descriptor: \"router\", \"fingerprint\", \"published\". Skipping descriptor.");
            return;
        }
        try {
            String scrubbedAddressString = this.scrubIpv4Address(address, fingerprintBytes, published);
            if (null == scrubbedAddressString) {
                logger.warn("Invalid IP address in \"router\" line in bridge server descriptor. Skipping descriptor.");
                return;
            }
            scrubbedAddress.append(scrubbedAddressString);
            for (Map.Entry e : scrubbedIpAddressesAndTcpPorts.entrySet()) {
                String scrubbedOrAddress = this.scrubOrAddress((String)e.getValue(), fingerprintBytes, published);
                if (null == scrubbedOrAddress) {
                    logger.warn("Invalid IP address or TCP port in \"or-address\" line in bridge server descriptor. Skipping descriptor.");
                    return;
                }
                ((StringBuilder)e.getKey()).append(scrubbedOrAddress);
            }
            for (Map.Entry e : scrubbedTcpPorts.entrySet()) {
                String scrubbedTcpPort = this.scrubTcpPort((String)e.getValue(), fingerprintBytes, published);
                if (null == scrubbedTcpPort) {
                    logger.warn("Invalid TCP port in \"router\" line in bridge server descriptor. Skipping descriptor.");
                    return;
                }
                ((StringBuilder)e.getKey()).append(scrubbedTcpPort);
            }
        }
        catch (IOException exception) {
            this.persistenceProblemWithSecrets = true;
            return;
        }
        String descriptorDigest = null;
        String ascii = new String(data, StandardCharsets.US_ASCII);
        String startToken = "router ";
        String sigToken = "\nrouter-signature\n";
        int start = ascii.indexOf(startToken);
        int sig = ascii.indexOf(sigToken) + sigToken.length();
        if (start >= 0 && sig >= 0 && sig > start) {
            byte[] forDigest = new byte[sig - start];
            System.arraycopy(data, start, forDigest, 0, sig - start);
            descriptorDigest = DigestUtils.sha1Hex(DigestUtils.sha1(forDigest));
        }
        if (descriptorDigest == null) {
            logger.warn("Could not calculate server descriptor digest.");
            return;
        }
        String descriptorDigestSha256Base64 = null;
        if (masterKeyEd25519FromIdentityEd25519 != null) {
            ascii = new String(data, StandardCharsets.US_ASCII);
            startToken = "router ";
            sigToken = "\n-----END SIGNATURE-----\n";
            start = ascii.indexOf(startToken);
            sig = ascii.indexOf(sigToken) + sigToken.length();
            if (start >= 0 && sig >= 0 && sig > start) {
                byte[] forDigest = new byte[sig - start];
                System.arraycopy(data, start, forDigest, 0, sig - start);
                descriptorDigestSha256Base64 = Base64.encodeBase64String(DigestUtils.sha256(DigestUtils.sha256(forDigest))).replaceAll("=", "");
            }
            if (descriptorDigestSha256Base64 == null) {
                logger.warn("Could not calculate server descriptor SHA256 digest.");
                return;
            }
        }
        if (null != descriptorDigestSha256Base64) {
            scrubbed.append("router-digest-sha256 ").append(descriptorDigestSha256Base64).newLine();
        }
        scrubbed.append("router-digest ").append(descriptorDigest.toUpperCase()).newLine();
        String dyear = published.substring(0, 4);
        String dmonth = published.substring(5, 7);
        File tarballFile = new File(this.sanitizedBridgesDirectory.getAbsolutePath() + "/" + dyear + "/" + dmonth + "/server-descriptors//" + descriptorDigest.charAt(0) + "/" + descriptorDigest.charAt(1) + "/" + descriptorDigest);
        try {
            File rsyncCatFile = new File(this.config.getPath(Key.RecentPath).toFile(), "bridge-descriptors/server-descriptors/" + this.rsyncCatString + "-server-descriptors.tmp");
            File[] outputFiles = new File[]{tarballFile, rsyncCatFile};
            boolean[] append = new boolean[]{false, true};
            for (int i = 0; i < outputFiles.length; ++i) {
                File outputFile = outputFiles[i];
                boolean appendToFile = append[i];
                if (!outputFile.exists() || appendToFile) {
                    outputFile.getParentFile().mkdirs();
                    BufferedWriter bw = new BufferedWriter(new FileWriter(outputFile, appendToFile));
                    bw.write(scrubbed.toString());
                    bw.close();
                    continue;
                }
                break;
            }
        }
        catch (IOException | ConfigurationException e) {
            logger.warn("Could not write sanitized server descriptor to disk.", e);
        }
    }

    private String parseMasterKeyEd25519FromIdentityEd25519(String identityEd25519Base64) {
        byte[] identityEd25519 = Base64.decodeBase64(identityEd25519Base64);
        if (identityEd25519.length < 40) {
            logger.warn("Invalid length of identity-ed25519 (in bytes): {}", (Object)identityEd25519.length);
        } else if (identityEd25519[0] != 1) {
            logger.warn("Unknown version in identity-ed25519: {}", (Object)identityEd25519[0]);
        } else if (identityEd25519[1] != 4) {
            logger.warn("Unknown cert type in identity-ed25519: {}", (Object)identityEd25519[1]);
        } else if (identityEd25519[6] != 1) {
            logger.warn("Unknown certified key type in identity-ed25519: {}", (Object)identityEd25519[1]);
        } else if (identityEd25519[39] == 0) {
            logger.warn("No extensions in identity-ed25519 (which would contain the encoded master-key-ed25519): {}", (Object)identityEd25519[39]);
        } else {
            int extensionStart = 40;
            for (int i = 0; i < identityEd25519[39]; ++i) {
                if (identityEd25519.length < extensionStart + 4) {
                    logger.warn("Invalid extension with id {} in identity-ed25519.", (Object)i);
                    break;
                }
                int extensionLength = identityEd25519[extensionStart];
                extensionLength <<= 8;
                byte extensionType = identityEd25519[extensionStart + 2];
                if ((extensionLength += identityEd25519[extensionStart + 1]) == 32 && extensionType == 4) {
                    if (identityEd25519.length < extensionStart + 4 + 32) {
                        logger.warn("Invalid extension with id {} in identity-ed25519.", (Object)i);
                        break;
                    }
                    byte[] masterKeyEd25519 = new byte[32];
                    System.arraycopy(identityEd25519, extensionStart + 4, masterKeyEd25519, 0, masterKeyEd25519.length);
                    String masterKeyEd25519Base64 = Base64.encodeBase64String(masterKeyEd25519);
                    return masterKeyEd25519Base64.replaceAll("=", "");
                }
                extensionStart += 4 + extensionLength;
            }
        }
        logger.warn("Unable to locate master-key-ed25519 in identity-ed25519.");
        return null;
    }

    public void sanitizeAndStoreExtraInfoDescriptor(byte[] data) {
        String scrubbedDesc = null;
        String published = null;
        String masterKeyEd25519FromIdentityEd25519 = null;
        try {
            String line;
            BufferedReader br = new BufferedReader(new StringReader(new String(data, "US-ASCII")));
            DescriptorBuilder scrubbed = null;
            String masterKeyEd25519 = null;
            while ((line = br.readLine()) != null) {
                String[] parts = line.split(" ");
                if (line.startsWith("extra-info ")) {
                    if (parts.length < 3) {
                        logger.debug("Illegal line in extra-info descriptor: '{}'.  Skipping descriptor.", (Object)line);
                        return;
                    }
                    String hashedBridgeIdentity = DigestUtils.sha1Hex(Hex.decodeHex(parts[2].toCharArray())).toLowerCase();
                    scrubbed = new DescriptorBuilder("extra-info ").append(parts[1]).space().append(hashedBridgeIdentity.toUpperCase()).newLine();
                    continue;
                }
                if (line.startsWith("published ")) {
                    scrubbed.append(line).newLine();
                    published = line.substring("published ".length());
                    if (published.compareTo(this.maxExtraInfoDescriptorPublishedTime) <= 0) continue;
                    this.maxExtraInfoDescriptorPublishedTime = published;
                    continue;
                }
                if (line.startsWith("transport ")) {
                    if (parts.length < 3) {
                        logger.debug("Illegal line in extra-info descriptor: '{}'.  Skipping descriptor.", (Object)line);
                        return;
                    }
                    scrubbed.append("transport ").append(parts[1]).newLine();
                    continue;
                }
                if (line.startsWith("transport-info ")) continue;
                if (line.equals("identity-ed25519")) {
                    StringBuilder sb = new StringBuilder();
                    while ((line = br.readLine()) != null && !line.equals("-----END ED25519 CERT-----")) {
                        if (line.equals("-----BEGIN ED25519 CERT-----")) continue;
                        sb.append(line);
                    }
                    masterKeyEd25519FromIdentityEd25519 = this.parseMasterKeyEd25519FromIdentityEd25519(sb.toString());
                    String sha256MasterKeyEd25519 = Base64.encodeBase64String(DigestUtils.sha256(Base64.decodeBase64(masterKeyEd25519FromIdentityEd25519 + "="))).replaceAll("=", "");
                    scrubbed.append("master-key-ed25519 ").append(sha256MasterKeyEd25519).newLine();
                    if (masterKeyEd25519 == null || masterKeyEd25519.equals(masterKeyEd25519FromIdentityEd25519)) continue;
                    logger.warn("Mismatch between identity-ed25519 and master-key-ed25519.  Skipping.");
                    return;
                }
                if (line.startsWith("master-key-ed25519 ")) {
                    masterKeyEd25519 = line.substring(line.indexOf(" ") + 1);
                    if (masterKeyEd25519FromIdentityEd25519 == null || masterKeyEd25519FromIdentityEd25519.equals(masterKeyEd25519)) continue;
                    logger.warn("Mismatch between identity-ed25519 and master-key-ed25519.  Skipping.");
                    return;
                }
                if (line.startsWith("write-history ") || line.startsWith("read-history ") || line.startsWith("geoip-start-time ") || line.startsWith("geoip-client-origins ") || line.startsWith("geoip-db-digest ") || line.startsWith("geoip6-db-digest ") || line.startsWith("conn-bi-direct ") || line.startsWith("bridge-") || line.startsWith("dirreq-") || line.startsWith("cell-") || line.startsWith("entry-") || line.startsWith("exit-") || line.startsWith("hidserv-") || line.startsWith("padding-counts ")) {
                    scrubbed.append(line).newLine();
                    continue;
                }
                if (line.startsWith("router-signature")) {
                    scrubbedDesc = scrubbed.toString();
                    break;
                }
                if (line.startsWith("router-sig-ed25519 ")) continue;
                logger.warn("Unrecognized line '{}'. Skipping.", (Object)line);
                return;
            }
            br.close();
        }
        catch (IOException | DecoderException e) {
            logger.warn("Could not parse extra-info descriptor.", e);
            return;
        }
        String descriptorDigest = null;
        String ascii = new String(data, StandardCharsets.US_ASCII);
        String startToken = "extra-info ";
        String sigToken = "\nrouter-signature\n";
        int start = ascii.indexOf(startToken);
        int sig = ascii.indexOf(sigToken) + sigToken.length();
        if (start >= 0 && sig >= 0 && sig > start) {
            byte[] forDigest = new byte[sig - start];
            System.arraycopy(data, start, forDigest, 0, sig - start);
            descriptorDigest = DigestUtils.sha1Hex(DigestUtils.sha1(forDigest));
        }
        if (descriptorDigest == null) {
            logger.warn("Could not calculate extra-info descriptor digest.");
            return;
        }
        String descriptorDigestSha256Base64 = null;
        if (masterKeyEd25519FromIdentityEd25519 != null) {
            ascii = new String(data, StandardCharsets.US_ASCII);
            startToken = "extra-info ";
            sigToken = "\n-----END SIGNATURE-----\n";
            start = ascii.indexOf(startToken);
            sig = ascii.indexOf(sigToken) + sigToken.length();
            if (start >= 0 && sig >= 0 && sig > start) {
                byte[] forDigest = new byte[sig - start];
                System.arraycopy(data, start, forDigest, 0, sig - start);
                descriptorDigestSha256Base64 = Base64.encodeBase64String(DigestUtils.sha256(DigestUtils.sha256(forDigest))).replaceAll("=", "");
            }
            if (descriptorDigestSha256Base64 == null) {
                logger.warn("Could not calculate extra-info descriptor SHA256 digest.");
                return;
            }
        }
        String dyear = published.substring(0, 4);
        String dmonth = published.substring(5, 7);
        File tarballFile = new File(this.sanitizedBridgesDirectory.getAbsolutePath() + "/" + dyear + "/" + dmonth + "/extra-infos/" + descriptorDigest.charAt(0) + "/" + descriptorDigest.charAt(1) + "/" + descriptorDigest);
        try {
            File rsyncCatFile = new File(this.config.getPath(Key.RecentPath).toFile(), "bridge-descriptors/extra-infos/" + this.rsyncCatString + "-extra-infos.tmp");
            File[] outputFiles = new File[]{tarballFile, rsyncCatFile};
            boolean[] append = new boolean[]{false, true};
            for (int i = 0; i < outputFiles.length; ++i) {
                File outputFile = outputFiles[i];
                boolean appendToFile = append[i];
                if (!outputFile.exists() || appendToFile) {
                    outputFile.getParentFile().mkdirs();
                    BufferedWriter bw = new BufferedWriter(new FileWriter(outputFile, appendToFile));
                    bw.write(Annotation.BridgeExtraInfo.toString());
                    bw.write(scrubbedDesc);
                    if (descriptorDigestSha256Base64 != null) {
                        bw.write("router-digest-sha256 " + descriptorDigestSha256Base64 + "\n");
                    }
                    bw.write("router-digest " + descriptorDigest.toUpperCase() + "\n");
                    bw.close();
                    continue;
                }
                break;
            }
        }
        catch (Exception e) {
            logger.warn("Could not write sanitized extra-info descriptor to disk.", e);
        }
    }

    public void finishWriting() {
        if (!this.secretsForHashingIpAddresses.isEmpty() && this.secretsForHashingIpAddresses.firstKey().compareTo(this.bridgeSanitizingCutOffTimestamp) < 0) {
            try {
                int kept = 0;
                int deleted = 0;
                BufferedWriter bw = new BufferedWriter(new FileWriter(this.bridgeIpSecretsFile));
                for (Map.Entry<String, byte[]> e : this.secretsForHashingIpAddresses.entrySet()) {
                    if (e.getKey().compareTo(this.bridgeSanitizingCutOffTimestamp) < 0) {
                        ++deleted;
                        continue;
                    }
                    bw.write(e.getKey() + "," + Hex.encodeHexString(e.getValue()) + "\n");
                    ++kept;
                }
                bw.close();
                logger.info("Deleted {} secrets that we don't need anymore and kept {}.", (Object)deleted, (Object)kept);
            }
            catch (IOException e) {
                logger.warn("Could not store reduced set of secrets to disk! This is a bad sign, better check what's going on!", e);
            }
        }
    }

    private void checkStaleDescriptors() {
        SimpleDateFormat dateTimeFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        dateTimeFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        long tooOldMillis = System.currentTimeMillis() - 19800000L;
        try {
            long maxExtraInfoDescriptorPublishedMillis;
            long maxServerDescriptorPublishedMillis;
            long maxNetworkStatusPublishedMillis = dateTimeFormat.parse(this.maxNetworkStatusPublishedTime).getTime();
            if (maxNetworkStatusPublishedMillis > 0L && maxNetworkStatusPublishedMillis < tooOldMillis) {
                logger.warn("The last known bridge network status was published {}, which is more than 5:30 hours in the past.", (Object)this.maxNetworkStatusPublishedTime);
            }
            if ((maxServerDescriptorPublishedMillis = dateTimeFormat.parse(this.maxServerDescriptorPublishedTime).getTime()) > 0L && maxServerDescriptorPublishedMillis < tooOldMillis) {
                logger.warn("The last known bridge server descriptor was published {}, which is more than 5:30 hours in the past.", (Object)this.maxServerDescriptorPublishedTime);
            }
            if ((maxExtraInfoDescriptorPublishedMillis = dateTimeFormat.parse(this.maxExtraInfoDescriptorPublishedTime).getTime()) > 0L && maxExtraInfoDescriptorPublishedMillis < tooOldMillis) {
                logger.warn("The last known bridge extra-info descriptor was published {}, which is more than 5:30 hours in the past.", (Object)this.maxExtraInfoDescriptorPublishedTime);
            }
        }
        catch (ParseException e) {
            logger.warn("Unable to parse timestamp for stale check.", e);
        }
    }

    public void cleanUpRsyncDirectory() throws ConfigurationException {
        long cutOffMillis = System.currentTimeMillis() - 259200000L;
        Stack<File> allFiles = new Stack<File>();
        allFiles.add(new File(this.config.getPath(Key.RecentPath).toFile(), BRIDGE_DESCRIPTORS));
        while (!allFiles.isEmpty()) {
            File file = (File)allFiles.pop();
            if (file.isDirectory()) {
                allFiles.addAll(Arrays.asList(file.listFiles()));
                continue;
            }
            if (file.lastModified() < cutOffMillis) {
                file.delete();
                continue;
            }
            if (!file.getName().endsWith(".tmp")) continue;
            file.renameTo(new File(file.getParentFile(), file.getName().substring(0, file.getName().lastIndexOf(".tmp"))));
        }
    }
}

