/* NVTV actions -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * Contents: X actions. Requires HAVE_X
 *
 */

#include "local.h" /* before everything else */

#ifndef HAVE_X
#error "actions.c requires X"
#endif

#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <X11/extensions/xf86vmode.h>

#include "actions.h"

/* -------- X Actions -------- */

/* Minimum vidmode extension version required */
#define MINMAJOR 0
#define MINMINOR 5

inline int sqr (int arg)
{
  return arg * arg;
}

Bool has_vidmode (Display *display)
{
  static int vidmode = 0;

  int MajorVersion, MinorVersion;
  int EventBase, ErrorBase;

  while (vidmode == 0) {
    if (!XF86VidModeQueryVersion (display, &MajorVersion, &MinorVersion))
    {
      fprintf(stderr, "Unable to query video extension version\n");
      vidmode = -1;
      break;
    }
    if (!XF86VidModeQueryExtension(display, &EventBase, &ErrorBase)) {
      fprintf(stderr, "Unable to query video extension information\n");
      vidmode = -1;
      break;
    }
    /* Fail if the extension version in the server is too old */
    if (MajorVersion < MINMAJOR || 
	(MajorVersion == MINMAJOR && MinorVersion < MINMINOR)) {
      fprintf(stderr,
	      "Xserver is running an old XFree86-VidModeExtension version"
	      " (%d.%d)\n", MajorVersion, MinorVersion);
      fprintf(stderr, "Minimum required version is %d.%d\n",
	      MINMAJOR, MINMINOR);
      vidmode = -1;
      break;
    }
    vidmode = 1;
    break;
  }
  if (vidmode > 0) return TRUE; else return FALSE;
}

/* get all modelines only once, so we don't have to free them */

int vidmodelines (Display *display, int screen, XF86VidModeModeInfo ***info)
{
  static XF86VidModeModeInfo **modesinfo = NULL; 
  static int modecount = 0;

  if (!modesinfo) {
    XF86VidModeGetAllModeLines (display, screen, &modecount, &modesinfo);
  }
  *info = modesinfo;
  return modecount;
}

void switch_vidmode (Display *display, int screen, int res_x, int res_y)
{
  XF86VidModeModeInfo **modesinfo;
  int modecount;
  int i;
  int dist;
  int best_dist = -1;
  int best_index = -1;

  RAISE (MSG_DEBUG, "vidmode %i %i", res_x, res_y);
  if (!has_vidmode (display)) return;
  modecount = vidmodelines (display, screen, &modesinfo);
  for (i = 0; i < modecount; i++) {
    dist = sqr(modesinfo [i]->hdisplay - res_x) + 
           sqr(modesinfo [i]->vdisplay - res_y);
    if (modesinfo [i]->hdisplay < res_x ||
        modesinfo [i]->vdisplay < res_y) {
      dist = -1;
    }
    if (dist >= 0 && (dist < best_dist || best_dist == -1)) {
      best_dist = dist;
      best_index = i;
    }
  }
  if (best_dist >= 0) {
    RAISE (MSG_DEBUG, "vidmode switch %i", best_index);
    XF86VidModeSwitchToMode (display, screen, modesinfo [best_index]);
  }
}

/*
 * Find X video mode with given resolution
 */

Bool find_vidmode (Display *display, int screen, int res_x, int res_y, 
  TVCrtcRegs *crt, make_vidmode make)
{
  XF86VidModeModeInfo **modesinfo;
  int modecount;
  int i;

  if (!has_vidmode (display)) return FALSE;
  modecount = vidmodelines (display, screen, &modesinfo);
  for (i = 0; i < modecount; i++) {
    if (modesinfo [i]->hdisplay == res_x && modesinfo [i]->vdisplay == res_y) 
      break;
  }
  if (i >= modecount) return FALSE;
  make (modesinfo [i]->hdisplay, modesinfo [i]->hsyncstart, 
	modesinfo [i]->hsyncend, modesinfo [i]->htotal,    
	modesinfo [i]->vdisplay, modesinfo [i]->vsyncstart,
	modesinfo [i]->vsyncend, modesinfo [i]->vtotal, 
	modesinfo [i]->dotclock, crt);
  return TRUE;
}

/*
 * get current X video mode
 */

Bool get_vidmode (Display *display, int screen, int *res_x, int *res_y,
  TVCrtcRegs *crt, make_vidmode make)
{
  XF86VidModeModeLine modeline;
  int dotclock;

  if (!has_vidmode (display)) return FALSE;
  XF86VidModeGetModeLine (display, screen, &dotclock, &modeline);
  if (res_x) *res_x = modeline.hdisplay;
  if (res_y) *res_y = modeline.vdisplay;
  make (modeline.hdisplay, modeline.hsyncstart, 
	modeline.hsyncend, modeline.htotal,    
	modeline.vdisplay, modeline.vsyncstart,
	modeline.vsyncend, modeline.vtotal, 
	dotclock, crt);
  return TRUE;
}

void center_window (Display *display, int screen, Window window,
  int res_x, int res_y)
{
  Window root;
  int x, y;
  unsigned width, height, borderw, depth;
  unsigned disp_w, disp_h;
  int screen_w, screen_h;
  int rx, ry;
  Window junkwin;
  int junkint;
  XF86VidModeModeLine modeline;

  RAISE (MSG_DEBUG, "center 0x%lx %i,%i", window, res_x, res_y);
  if (window == None || !has_vidmode (display)) return;
  XGetGeometry (display, window, &root, &x, &y, &width, &height, 
		&borderw, &depth);
  (void) XTranslateCoordinates (display, window, root, 
				0, 0, &rx, &ry, &junkwin);
  if (res_x >= 0 && res_y >= 0) {
    disp_w = res_x;
    disp_h = res_y;
  } else {
    XF86VidModeGetModeLine (display, screen, &junkint, &modeline);
    if (modeline.privsize > 0) XFree(modeline.private);
    disp_w = modeline.hdisplay;
    disp_h = modeline.vdisplay;
  }
  screen_w = DisplayWidth (display, screen);
  screen_h = DisplayHeight (display, screen);
  x = rx + ((int) width - (int) disp_w) / 2;
  y = ry + ((int) height - (int) disp_h) / 2;
  if (x < 0) x = 0;
  if (y < 0) y = 0;
  if (x > screen_w) x = screen_w;
  if (y > screen_h) y = screen_h;
  XF86VidModeSetViewPort (display, screen, x, y);
  XWarpPointer (display, None, window, 0, 0, 0, 0, 
		width / 2, height / 2);
  /* FIXME TODO: Better, disable the pointer by setting an empty pixmap
     or grabbing it, and restore image on leave event or on the next
     mouse event. Will work only in interactive (gui) mode! */
}

/* -------- dsimple.c -------- */

/* This is are (slightly modified) routines from programs/xlsfonts/dsimple.c 
 * in the XFree sources, written by Mark Lillibridge.
 */

#define Fatal_Error(X) { fprintf(stderr, X); exit (1); }

Window Select_Window (Display *dpy, int screen)
{
  int status;
  Cursor cursor;
  XEvent event;
  Window target_win = None, root = RootWindow(dpy,screen);
  int buttons = 0;

  /* Make the target cursor */
  cursor = XCreateFontCursor(dpy, XC_crosshair);

  /* Grab the pointer using target cursor, letting it room all over */
  status = XGrabPointer(dpy, root, False,
			ButtonPressMask|ButtonReleaseMask, GrabModeSync,
			GrabModeAsync, root, cursor, CurrentTime);
  if (status != GrabSuccess) Fatal_Error("Can't grab the mouse.");

  /* Let the user select a window... */
  while ((target_win == None) || (buttons != 0)) {
    /* allow one more event */
    XAllowEvents(dpy, SyncPointer, CurrentTime);
    XWindowEvent(dpy, root, ButtonPressMask|ButtonReleaseMask, &event);
    switch (event.type) {
    case ButtonPress:
      if (target_win == None) {
	target_win = event.xbutton.subwindow; /* window selected */
	if (target_win == None) target_win = root;
      }
      buttons++;
      break;
    case ButtonRelease:
      if (buttons > 0) /* there may have been some down before we started */
	buttons--;
       break;
    }
  } 
  XUngrabPointer(dpy, CurrentTime);      /* Done with pointer */
  return(target_win);
}

/*
 * Window_With_Name: routine to locate a window with a given name on a display.
 *                   If no window with the given name is found, 0 is returned.
 *                   If more than one window has the given name, the first
 *                   one found will be returned.  Only top and its subwindows
 *                   are looked at.  Normally, top should be the RootWindow.
 */

Window Window_With_Name (Display *dpy, Window top, char *name)
{
  Window *children, dummy;
  unsigned int nchildren;
  int i;
  Window w = None;
  char *window_name;

  if (XFetchName(dpy, top, &window_name) && !strcmp(window_name, name))
    return(top);

  if (!XQueryTree(dpy, top, &dummy, &dummy, &children, &nchildren))
    return None;

  for (i=0; i<nchildren; i++) {
    w = Window_With_Name(dpy, children[i], name);
    if (w != None)
      break;
  }
  if (children) XFree ((char *)children);
  return w;
}

/*

TODO: RandR extension

see xrandr.c

XRRScreenSize *sizes;
XRRScreenConfiguration *sc;
int nsize;
int           nrate;
short         *rates;

sc = XRRGetScreenInfo (dpy, root);
sizes = XRRConfigSizes(sc, &nsize);

// rates = XRRConfigRates (sc, i, &nrate);

XRRSetScreenConfigAndRate (dpy, sc, DefaultRootWindow (dpy), 
 (SizeID) size, (Rotation) (rotation | reflection), rate, CurrentTime);

// wait for config notify event on root window??

XRRFreeScreenConfigInfo(sc);


*/
