# Sim::AgentSoar

Sim::AgentSoar is a SOAR-inspired explicit search architecture with a pluggable Large Language Model (LLM) worker.

It separates deterministic search control from heuristic proposal, allowing a local LLM (via Ollama) to guide operator selection without compromising structural integrity.

---

## Overview

Sim::AgentSoar implements:

* Explicit state-space search
* Best-first ordering (metric, then depth)
* Configurable branching factor
* Optional bounded regression correction
* Deterministic state validation
* Instrumented runtime statistics
* Pluggable LLM proposal layer

The included domain is a simple integer reachability problem used for calibration and experimentation.

---

## Architecture

The system is divided into:

* **Engine** — deterministic environment logic
* **Search** — explicit search controller
* **Worker** — LLM-backed operator proposer
* **Node** — structured search state

The LLM never evaluates states or controls search order.
It only proposes candidate operators.

All correctness guarantees remain deterministic.

---

## Requirements

* Perl 5.14+
* Ollama installed locally
* A local model (default: `llama3.2:1b`)

The Ollama daemon must be running:

```
ollama serve
```

---

## Example

```perl
use Sim::AgentSoar::AgentSoar;
use Sim::AgentSoar::Worker;

my $worker = Sim::AgentSoar::Worker->new(
    model => 'llama3.2:1b',
);

my $search = Sim::AgentSoar::AgentSoar->new(
    worker => $worker,
    branching_factor => 2,
    regression_tolerance => 2,
);

my $path = $search->run(
    start  => 4,
    target => 19,
);

if ($path) {
    print "Solution found\n";
}
```

---

## Design Philosophy

Sim::AgentSoar is built around three principles:

1. Structural recursion belongs in the search tree.
2. Heuristic reasoning belongs in the LLM.
3. Determinism must never depend on the LLM.

This separation prevents heuristic instability from corrupting the search backbone.

---

## License

GPL v3

---

Author: Gian Luca Brunetti, 2026 - gianluca.brunetti@gmail.com

