/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.rest.debug;

import static org.apache.juneau.commons.utils.CollectionUtils.*;
import static org.apache.juneau.commons.utils.StringUtils.*;
import static org.apache.juneau.commons.utils.Utils.*;
import static org.apache.juneau.rest.annotation.RestOpAnnotation.*;

import java.util.*;
import org.apache.juneau.*;
import org.apache.juneau.cp.*;
import org.apache.juneau.commons.reflect.*;
import org.apache.juneau.commons.utils.*;
import org.apache.juneau.rest.*;
import org.apache.juneau.rest.annotation.*;
import org.apache.juneau.rest.util.*;
import org.apache.juneau.svl.*;

/**
 * Default implementation of the {@link DebugEnablement} interface.
 *
 * <p>
 * Enables debug mode based on the following annotations:
 * <ul>
 * 	<li class='ja'>{@link Rest#debug()}
 * 	<li class='ja'>{@link RestOp#debug()}
 * 	<li class='ja'>{@link Rest#debugOn()}
 * </ul>
 *
 * <h5 class='section'>See Also:</h5><ul>
 * 	<li class='link'><a class="doclink" href="https://juneau.apache.org/docs/topics/RestServerLoggingAndDebugging">Logging / Debugging</a>
 * </ul>
 */
public class BasicDebugEnablement extends DebugEnablement {

	private static final AnnotationProvider AP = AnnotationProvider.INSTANCE;

	/**
	 * Constructor.
	 *
	 * @param beanStore The bean store containing injectable beans for this enablement.
	 */
	public BasicDebugEnablement(BeanStore beanStore) {
		super(beanStore);
	}

	@Override
	protected Builder init(BeanStore beanStore) {
		var b = super.init(beanStore);

		var defaultSettings = beanStore.getBean(DefaultSettingsMap.class).get();
		var builder = beanStore.getBean(RestContext.Builder.class).get();
		var resource = beanStore.getBean(ResourceSupplier.class).get();
		var varResolver = beanStore.getBean(VarResolver.class).get();
		var ap = AP;

		// Default debug enablement if not overridden at class/method level.
		var debugDefault = defaultSettings.get(Enablement.class, "RestContext.debugDefault").orElse(builder.isDebug() ? Enablement.ALWAYS : Enablement.NEVER);
		b.defaultEnable(debugDefault);

		var ci = ClassInfo.ofProxy(resource.get());

		// Gather @Rest(debug) settings.
		// @formatter:off
		rstream(ap.find(Rest.class, ci)).map(AnnotationInfo::inner).forEach(x -> {
			var x2 = varResolver.resolve(x.debug());
			if (! x2.isEmpty())
				b.enable(Enablement.fromString(x2), ci.getNameFull());
		});
		// @formatter:on

		// Gather @RestOp(debug) settings.
		// @formatter:off
		ci.getPublicMethods().stream()
			.forEach(x ->
				rstream(ap.find(x))
					.filter(REST_OP_GROUP)
					.flatMap(ai -> ai.getValue(String.class, "debug").stream())
					.filter(Utils::ne)
					.map(varResolver::resolve)
					.map(Enablement::fromString)
					.filter(Objects::nonNull)
					.forEach(e -> b.enable(e, x.getFullName()))
			);
		// @formatter:on

		// Gather @Rest(debugOn) settings.
		// @formatter:off
		rstream(ap.find(Rest.class, ci)).map(AnnotationInfo::inner).forEach(x -> {
			var x2 = varResolver.resolve(x.debugOn());
			for (var e : splitMap(x2, true).entrySet()) {
				var k = e.getKey();
				var v = e.getValue();
				if (v.isEmpty())
					v = "ALWAYS";
				if (! k.isEmpty())
					opt(Enablement.fromString(v)).ifPresent(en -> b.enable(en, k));
			}
		});
		// @formatter:on

		return b;
	}
}